<?php

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;
use PrestaShop\PrestaShop\Core\Grid\Action\Bulk\Type\SubmitBulkAction;

require dirname(__FILE__).'/libs/Maksekeskus.php';
require dirname(__FILE__).'/classes/Method.php';

if(file_exists(_PS_MODULE_DIR_ .'makecommerceomniva/makecommerceomniva.php'))
    include_once( _PS_MODULE_DIR_ .'makecommerceomniva/makecommerceomniva.php');
if(file_exists(_PS_MODULE_DIR_ .'makecommercesmartpost/makecommercesmartpost.php'))
    include_once( _PS_MODULE_DIR_ .'makecommercesmartpost/makecommercesmartpost.php');
if(file_exists(_PS_MODULE_DIR_ .'makecommercedpd/makecommercedpd.php'))
    include_once( _PS_MODULE_DIR_ .'makecommercedpd/makecommercedpd.php');
if(file_exists(_PS_MODULE_DIR_ .'makecommercelpexpress/makecommercelpexpress.php'))
    include_once( _PS_MODULE_DIR_ .'makecommercelpexpress/makecommercelpexpress.php');

class MakeCommerce extends PaymentModule
{
    const MODULE_VERSION = '3.4.0';
    const MODULE_NAME = 'makecommerce';

    const STATUS_NAME = 'OS_MK_BILLING';


    const TYPE_BANK = 'banklinks';
    const TYPE_CARD = 'cards';
    const TYPE_OTHER = 'other';
    const TYPE_PAYLATER = 'payLater';

    const TEST = 0;
    const LIVE = 1;

    const DISPLAY_ORDER_PAGE = 1;
    const DISPLAY_WIDGET = 2;
    const IMG_SIZE_S = 's';
    const IMG_SIZE_M = 'm';
    const IMG_SIZE_L = 'l';

    const A4 = 'A4';
    const A4_FULL_PAGE = 'A4_FULL_PAGE';
    const A5 = 'A5';
    const A6_FULL_PAGE = 'A6_FULL_PAGE';

    const OMNIVA = 'OMNIVA';
    const OMNIVA_LV = 'OMNIVA_LV';
    const OMNIVA_LT = 'OMNIVA_LT';

    const DPD_EE = 'DPD';
    const DPD_LV = 'DPD_LV';
    const DPD_LT = 'DPD_LT';

    const CACHE_VALID_TIME = 3600;

    const COUNTRY_ALL = 'all';

    const LOG_DEBUG = 'makecommerce-debug.log';

    private $fields = array(
        'secret_key',
        'shop_id',
        'publishable_key',
        'secret_key_test',
        'shop_id_test',
        'publishable_key_test',
        'server',
        'methods_display',
        'methods_named',
        'credit_display',
        'prefill_form',
        'logo_size',
        'methods_order',
        'omniva',
        'omniva_use_mk_tms',
        'omniva_verify_tms',
        'omniva_username',
        'omniva_password',
        'omniva_sender_name',
        'omniva_phone',
        'omniva_email',
        'omniva_street',
        'omniva_city',
        'omniva_country',
        'omniva_countrysel',
        'omniva_zip',
        'smartpost',
        'smartpost_apikey',
        'smartpost_sender_name',
        'smartpost_phone',
        'smartpost_email',
        'smartpost_city',
        'smartpost_street',
        'smartpost_country',
        'smartpost_zip',
        'dpd',
        'dpd_use_mk_tms',
        'dpd_username',
        'dpd_password',
        'dpd_apiKey',
        'dpd_countrysel',
        'dpd_sender_name',
        'dpd_phone',
        'dpd_email',
        'dpd_city',
        'dpd_street',
        'dpd_country',
        'dpd_zip',
        'lp_express_lt',
        'lp_express_lt_username',
        'lp_express_lt_password',
        'lp_express_lt_template',
        'lp_express_lt_sender_name',
        'lp_express_lt_phone',
        'lp_express_lt_email',
        'lp_express_lt_city',
        'lp_express_lt_street',
        'lp_express_lt_building',
        'lp_express_lt_zip',
        'log',
        'label_pgsize',
        'parcel_grouping'
    );

    private $lang_fields = array(
        'widget_title'
    );

    private $types = array(
        self::TYPE_BANK,
        self::TYPE_OTHER,
        self::TYPE_CARD,
        self::TYPE_PAYLATER
    );

    private static $api = null;

    private $html = '';

    public function __construct()
    {
        $this->name = self::MODULE_NAME;
        $this->tab = 'payments_gateways';
        $this->version = self::MODULE_VERSION;
        $this->need_instance = 0;
        $this->author = 'MakeCommerce.net';
        $this->bootstrap = true;
        $this->controllers = array('payment', 'validation', 'confirmation');

        parent::__construct();
        // $this->install(); // refresh hooks
        $this->displayName = 'MakeCommerce';
        $this->description = 'Payment Gateway by Maksekeskus AS';
    }

    public function installHooks() {
        if (
            !parent::install() OR
            !$this->registerHook('header') OR //
            // !$this->registerHook('adminOrder') OR //
            !$this->registerHook('displayBackOfficeHeader') OR //
            !$this->registerHook('paymentOptions') OR //
            !$this->registerHook('displayPaymentReturn') OR //
            !$this->registerHook('displayHeader') OR //
            !$this->registerHook('displayAdminOrder') OR
            !$this->registerHook('displayAdminOrderMain') OR
            !$this->registerHook('actionFrontControllerSetMedia') OR
            !$this->registerHook('actionOrderGridDefinitionModifier') OR
            !$this->registerHook('displayOrderDetail') OR
            !$this->createOrderState() OR
            !$this->installTitles()
        ) return false; else return true;
    }

    public function installHooksOld() {
        if (
            !parent::install() OR
            !$this->registerHook('header') OR
            !$this->registerHook('adminOrder') OR
            !$this->registerHook('backOfficeHeader') OR
            !$this->registerHook('paymentOptions') OR
            !$this->registerHook('displayPaymentReturn') OR
            !$this->registerHook('displayHeader') OR
            !$this->registerHook('displayOrderDetail') OR
            !$this->registerHook('actionFrontControllerSetMedia') OR
            !$this->createOrderState() OR
            !$this->installTitles()
        ) return false; else return true;
    }

    public function install()
    {
        $initStatus = false;
        if ( version_compare(_PS_VERSION_, "1.7.7.0", '<') ) {
            $initStatus = $this->installHooksOld();
        } else {  // register hooks for newer
            $initStatus = $this->installHooks();
        }
        if (!(Configuration::hasKey('parcel_grouping'))) {
            Configuration::updateValue('parcel_grouping',1);
        }
        if (!$initStatus) {
            return false;
        } else {
            $this->createTable();
            $this->downloadCarrierModules('makecommerceomniva');
            $this->downloadCarrierModules('makecommercesmartpost');
            $this->downloadCarrierModules('makecommercedpd');
            $this->downloadCarrierModules('makecommercelpexpress');
            return true;
        }
    }

    public function uninstall()
    {
        $carriers = [
            'makecommerceomniva' => 'MakeCommerceOmniva',
            'makecommercesmartpost' => 'MakeCommerceSmartpost',
            'makecommercedpd' => 'MakeCommerceDpd',
            'makecommercelpexpress' => 'MakeCommerceLpExpress'
        ];

        foreach ($carriers as $key => $class) {
            $carrier_folder = _PS_MODULE_DIR_ . $key;
            if (is_dir($carrier_folder) && class_exists($class)) {
                $carrier = new $class();
                $carrier->uninstall();
                $this->recursiveDeleteOnDisk($carrier_folder);
            }
        }

        return parent::uninstall();
    }

    public function createTable()
    {
        if (!Db::getInstance()->execute('
                CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'makecommerce_refunds` (
                  `id_makecommerce_refund` int(10) unsigned NOT NULL AUTO_INCREMENT,
                  `id_order` int(10) unsigned NOT NULL,
                  `refund_amount` decimal(20,6)	 NOT NULL,
                  `refund_date` datetime NOT NULL,
                  PRIMARY KEY (`id_makecommerce_refund`)
                  ) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=UTF8;
                ')) {
            return false;
        }
    }

    public function installTitles()
    {
        $languages = Language::getLanguages(false);
        foreach ($languages as $lang)
            $this->installTitle((int)$lang['id_lang']);

        return true;
    }

    public function downloadCarrierModules($name)
    {
        return Tools::ZipExtract(dirname(__FILE__) . "/src/${name}.zip", _PS_MODULE_DIR_);
    }

    protected function recursiveDeleteOnDisk($dir)
    {
        if (strpos(realpath($dir), realpath(_PS_MODULE_DIR_)) === false) {
            return;
        }
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != '.' && $object != '..') {
                    if (filetype($dir.'/'.$object) == 'dir') {
                        $this->recursiveDeleteOnDisk($dir.'/'.$object);
                    } else {
                        unlink($dir.'/'.$object);
                    }
                }
            }
            reset($objects);
            rmdir($dir);
        }
    }

    public function installTitle($id_lang)
    {
        $values[self::prefixed('widget_title')][(int)$id_lang] = 'Pay by banklink or card';
        Configuration::updateValue(self::prefixed('widget_title'), $values[self::prefixed('widget_title')]);
    }

    public function isUsingNewTranslationSystem()
    {
        return false;
    }

    public function getContent()
    {
        if (Tools::isSubmit('submit'.$this->name)) {
            foreach ($this->getCarriers() as $carrier) {
                $this->fields[] = "carrier_" . $carrier["id_carrier"];
            }

            foreach ($this->fields as $field) {
                if (in_array($field, array('omniva', 'smartpost', 'dpd', 'lp_express_lt'))) {
                    $this->handleCarrierModuleAction($field);
                }
                if ($field === 'omniva_use_mk_tms') {
                    $isEnabled = $this->validateTmsAvailability();
                    $this->updateConfig($field, !$isEnabled ? false : Tools::getValue($field));
                } else {
                    $this->updateConfig($field, Tools::getValue($field));
                }
            }

            foreach ($this->lang_fields as $param){
                $langvalues = array();
                foreach (Language::getLanguages(false) as $key => $language){
                    $langvalues[$language['id_lang']] = Tools::getValue($param.'_'.$language['id_lang']);
                }
                $this->updateConfig($param, $langvalues, true);
            }

            // delete carriers cache upon update
            $this->clearCarriersCache();
            $this->clearPaymentCache();

            $this->html .= $this->displayConfirmation($this->l('Settings updated from MakeCommerce servers'));
        }

        $this->html.=$this->showHeader();
        $this->showForm();
        return $this->html;
    }

    /**
     * Handles carriers modules install and uninstall actions
     */
    private function handleCarrierModuleAction($field)
    {
        // map carriers classes for uninstall/install handling
        $moduleClassMap = array(
            'omniva' => 'MakeCommerceOmniva',
            'smartpost' => 'MakeCommerceSmartPost',
            'dpd' => 'MakeCommerceDpd',
            'lp_express_lt' => 'MakeCommerceLpExpress',
        );

        if (!in_array($field, array_keys($moduleClassMap))) {
            return false;
        }

        if (Tools::getValue($field)) {
            $isInstalled = self::getConfig($field);

            if (!$isInstalled) {
                $api = $this->getApi();

                // check if class exists, if not extract the zip file and include it
                if (!class_exists($moduleClassMap[$field])) {
                    $carrierName = strtolower($moduleClassMap[$field]);
                    if ($this->downloadCarrierModules($carrierName)) {
                        include_once _PS_MODULE_DIR_ .$carrierName.'/'.$carrierName.'.php';
                    }
                }

                if (!is_null($api)) {
                    $module = new $moduleClassMap[$field]();
                    $module->install();
                    $this->html .= $this->displayConfirmation(
                        sprintf(
                            $this->l('"%s parcel terminals" carrier enabled'),
                            ucfirst($field)
                        )
                    );
                } else {
                    $this->html .= $this->displayError(
                        sprintf(
                            $this->l('Cannot enable "%s Parcel terminals" carrier. Save your Shop id, Secret key and Publishable key.'),
                            ucfirst($field)
                        )
                    );
                }
            }
        } else {
            $isInstalled = self::getConfig($field);
            if ($isInstalled && class_exists($moduleClassMap[$field])) {
                $module = new $moduleClassMap[$field]();
                $module->uninstall();
                $this->html .= $this->displayConfirmation(
                    sprintf(
                        $this->l('"%s Parcel Terminals carrier" disabled'),
                        ucfirst($field)
                    )
                );
            }
        }

    }

    private function showHeader()
    {
        $this->smarty->assign(
            array(
                'path' => $this->getPathUri()
            )
        );
        return $this->display(__FILE__, 'settings_header.tpl');
    }

    private function showForm()
    {
        $field_values = array();
        $default_lang = (int)Configuration::get('PS_LANG_DEFAULT');
        $helper = new HelperForm();

        $return = $this->validateTmsAvailability();

        $fields_form[0]['form'] = array(
            'legend' => array(
                'title' => $this->l('Settings'),
            ),
            'input' => array(
                array(
                    'type' => 'select',
                    'label' => $this->l('Server'),
                    'name' => 'server',
                    'desc'     => $this->l('For Test Server see more about: ').'<a href="https://maksekeskus.ee/en/for-developers/test-environment/">MakeCommerce Test environment</a>',
                    'lang' => false,
                    'options' => array(
                        'query' => array(
                            array('id'=> self::TEST, 'name' => $this->l('Test server')),
                            array('id'=> self::LIVE, 'name' => $this->l('Live server'))
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    )
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Shop id'),
                    'desc'     => $this->l('You get the Shop ID and API keys from our Merchant Portal after sign-up'),
                    'name' => 'shop_id',
                    'size' => 30,
                    'form_group_class' => 'live_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Secret key'),
                    'name' => 'secret_key',
                    'form_group_class' => 'live_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Publishable key'),
                    'name' =>  'publishable_key',
                    'form_group_class' => 'live_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Shop id'),
                    'name' => 'shop_id_test',
                    'form_group_class' => 'test_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Secret key'),
                    'name' => 'secret_key_test',
                    'form_group_class' => 'test_settings',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Publishable key'),
                    'name' =>  'publishable_key_test',
                    'form_group_class' => 'test_settings',
                    'required' => false,
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
            ),
        );

        $fields_form[2]['form'] = array(
            'legend' => array(
                'title' => $this->l('Design'),
            ),
            'input' => array(
                array(
                    'type' => 'select',
                    'label' => $this->l('Display payment channels as'),
                    'name' => 'methods_display',
                    'required' => false,
                    'options' => array(
                        'query' => array(
                            array(
                                'id'=> self::DISPLAY_WIDGET,
                                'name' => $this->l('Widget')
                            ),
                            array(
                                'id'=> self::DISPLAY_ORDER_PAGE,
                                'name' => $this->l('List')
                            ),
                        ),
                        'id' => 'id',
                        'name' => 'name',
                    ),
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Widget title'),
                    'name' =>  'widget_title',
                    'form_group_class' => 'widget_setting',
                    'lang' => true,
                    'required' => false,
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Size of channel logos'),
                    'form_group_class' => 'widget_setting',
                    'name' => 'logo_size',
                    'required' => false,
                    'options' => array(
                        'query' => array(
                            array(
                                'id'=> self::IMG_SIZE_S,
                                'name' => $this->l('Small'),
                            ),
                            array(
                                'id'=> self::IMG_SIZE_M,
                                'name' => $this->l('Medium'),
                            ),
                            array(
                                'id'=> self::IMG_SIZE_L,
                                'name' => $this->l('Large'),
                            ),
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    )
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Shipping label print format'),
                    'name' => 'label_pgsize',
                    'required' => false,
                    'options' => array(
                        'query' => array(
                            array(
                                'id'=> self::A4,
                                'name' => $this->l('A4'),
                            ),
                            array(
                                'id'=> self::A4_FULL_PAGE,
                                'name' => $this->l('A4 full page'),
                            ),
                            array(
                                'id'=> self::A5,
                                'name' => $this->l('A5'),
                            ),
                            array(
                                'id'=> self::A6_FULL_PAGE,
                                'name' => $this->l('A6 full page'),
                            ),
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    )
                ),

                array(
                    'type' => 'switch',
                    'label' => $this->l('Credit cards in separate group'),
                    'form_group_class' => 'widget_setting',
                    'name' => 'credit_display',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'separate_group_on',
                            'value' => 1,
                            'label' => $this->l('Yes'),
                        ),
                        array(
                            'id' => 'separate_group_off',
                            'value' => 0,
                            'label' => $this->l('No'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'label' => $this->l('Show payment method names')."*",
                    'form_group_class' => 'widget_setting',
                    'name' => 'methods_named',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'methods_named_on',
                            'value' => 1,
                            'label' => $this->l('Yes'),
                        ),
                        array(
                            'id' => 'methods_named_off',
                            'value' => 0,
                            'label' => $this->l('No'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'label' => $this->l('Parcel machines not grouped by location'),
                    'name' => 'parcel_grouping',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'pmgrouping_on',
                            'value' => 0,
                            'label' => $this->l('Yes'),
                        ),
                        array(
                            'id' => 'pmgrouping_off',
                            'value' => 1,
                            'label' => $this->l('No'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'label' => $this->l('Prefill credit card form with customer data'),
                    'name' => 'prefill_form',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'prefill_on',
                            'value' => 1,
                            'label' => $this->l('Yes')
                        ),
                        array(
                            'id' => 'prefill_off',
                            'value' => 0,
                            'label' => $this->l('No'),
                        )
                    )
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Payment methods order'),
                    'desc'     => $this->l('i.e.:  lhv, seb, swedbank'),
                    'name' =>  'methods_order',
                    'required' => false,
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
            ),
        );

        $return_countries = array();
        $countries = Country::getCountries($this->context->employee->id_lang);
        foreach ($countries as $country) {
            $return_countries[] = array('id'=> $country['iso_code'], 'name' => $country['country']);
        }

        $fields_form[3]['form'] = array(
            'legend' => array(
                'title' => $this->l('Omniva Parcel Terminals'),
            ),
            'input' => array(
                array(
                    'type' => 'switch',
                    'label' => $this->l('Enable Omniva Parcel Terminals carrier'),
                    'name' => 'omniva',
                    'form_group_class' => 'omniva_switch',
                    'required' => false,
                    'is_bool' => true,
                    'desc' => $this->l('If enabled the Omniva Parcel Terminals will appear under Shipping->Carriers. You can configure pricing there.').'<br>'.$this->l('Here you can configure integration with Omniva systems. See more about it on').' <a href="https://https://maksekeskus.ee/maksekeskuse-module-prestashopile/#omniva">Module home page</a> ',
                    'values' => array(
                        array(
                            'id' => 'omniva_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'omniva_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'label' => $this->l('Use Makecommerce Shipping+ service'),
                    'name' => 'omniva_use_mk_tms',
                    'form_group_class' => 'omniva_setting',
                    'required' => false,
                    'is_bool' => true,
                    'desc' => $this->l('If enabled uses Makecommerce transport mediation service to create a shipment') .
                        '<div class="alert alert-info" role="alert">
                            <p class="alert-text">'. $this->l('Please note that our Transport meditation service (Shipping+) is not currently supported for courier types. For courier services, you are required to have your own credentials.') .'</p>
                        </div>',
                    'values' => array(
                        array(
                            'id' => 'omniva_use_mk_tms_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'omniva_use_mk_tms_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Omniva WebServices Username'),
                    'name' =>  'omniva_username',
                    'form_group_class' => 'omniva_setting omniva_tms_setting',
                    'hint' => $this->l('You get it from your Account Manager in Omniva'),
                    'required' => false,
                    'size' => 30,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Password'),
                    'name' =>  'omniva_password',
                    'form_group_class' => 'omniva_setting omniva_tms_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Country for Omniva'),
                    'form_group_class' => 'omniva_setting',
                    'name' => 'omniva_countrysel',
                    'required' => false,
                    'options' => array(
                        'query' => array(
                            array(
                                'id'=> self::OMNIVA,
                                'name' => $this->l('Omniva EE'),
                            ),
                            array(
                                'id'=> self::OMNIVA_LV,
                                'name' => $this->l('Omniva LV'),
                            ),
                            array(
                                'id'=> self::OMNIVA_LT,
                                'name' => $this->l('Omniva LT'),
                            ),
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    )
                ),
                array('type' => 'hr', 'name'=> ''),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Name'),
                    'name' =>  'omniva_sender_name',
                    'form_group_class' => 'omniva_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Phone'),
                    'name' =>  'omniva_phone',
                    'form_group_class' => 'omniva_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Email'),
                    'name' =>  'omniva_email',
                    'form_group_class' => 'omniva_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - City'),
                    'name' =>  'omniva_city',
                    'form_group_class' => 'omniva_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Street'),
                    'name' =>  'omniva_street',
                    'form_group_class' => 'omniva_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Postal code'),
                    'name' =>  'omniva_zip',
                    'form_group_class' => 'omniva_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Return address - Country'),
                    'name' => 'omniva_country',
                    'lang' => false,
                    'form_group_class' => 'omniva_setting',
                    'options' => array(
                        'query' => $return_countries,
                        'id' => 'id',
                        'name' => 'name'
                    ),
                    'required' => false,
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
            ),
        );

        $fields_form[4]['form'] = array(
            'legend' => array(
                'title' => $this->l('Itella SmartPOST parcel terminals'),
            ),
            'input' => array(
                array(
                    'type' => 'switch',
                    'label' => $this->l('Enable Itella SmartPOST carrier'),
                    'name' => 'smartpost',
                    'form_group_class' => 'smartpost_switch',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'smartpost_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'smartpost_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                    'desc' => $this->l('If enabled the "Itella SmartPOST Parcel Terminals" will appear under Shipping->Carriers. You can configure pricing there.').'<br>'.$this->l('Here you can confgure integration with Itella systems. See more about it on').' <a href="https://https://maksekeskus.ee/maksekeskuse-module-prestashopile/#itella">Module home page</a> ',
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Api key'),
                    'name' =>  'smartpost_apikey',
                    'form_group_class' => 'smartpost_setting',
                    'required' => false,
                ),
                array('type' => 'hr', 'name'=> ''),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Name'),
                    'name' =>  'smartpost_sender_name',
                    'form_group_class' => 'smartpost_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Phone'),
                    'name' =>  'smartpost_phone',
                    'form_group_class' => 'smartpost_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Email'),
                    'name' =>  'smartpost_email',
                    'form_group_class' => 'smartpost_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - City'),
                    'name' =>  'smartpost_city',
                    'form_group_class' => 'smartpost_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Postal code'),
                    'name' =>  'smartpost_zip',
                    'form_group_class' => 'smartpost_setting',
                    'required' => false,
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
            ),
        );

        $fields_form[5]['form'] = array(
            'legend' => array(
                'title' => $this->l('DPD parcel terminals'),
            ),
            'input' => array(
                array(
                    'type' => 'switch',
                    'label' => $this->l('Enable DPD carrier'),
                    'name' => 'dpd',
                    'form_group_class' => 'dpd_switch',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'dpd_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'dpd_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                    'desc' => $this->l('If enabled the "DPD Parcel Terminals" will appear under Shipping->Carriers. You can configure pricing there.'),
                ),
                array(
                    'type' => 'switch',
                    'label' => $this->l('Use Makecommerce Shipping+ service'),
                    'name' => 'dpd_use_mk_tms',
                    'form_group_class' => 'dpd_setting',
                    'required' => false,
                    'is_bool' => true,
                    'desc' => $this->l('If enabled uses Makecommerce transport mediation service to create a shipment') .
                        '<div class="alert alert-info" role="alert">
                            <p class="alert-text">'. $this->l('Please note that our Transport meditation service (Shipping+) is not currently supported for courier types. For courier services, you are required to have your own credentials.') .'</p>
                        </div>',
                    'values' => array(
                        array(
                            'id' => 'dpd_use_mk_tms_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'dpd_use_mk_tms_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Username'),
                    'name' =>  'dpd_username',
                    'hint' => $this->l('You get after sign up on web service'),
                    'form_group_class' => 'dpd_setting dpd_tms_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Password'),
                    'name' =>  'dpd_password',
                    'form_group_class' => 'dpd_setting dpd_tms_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Api key'),
                    'name' =>  'dpd_apiKey',
                    'form_group_class' => 'dpd_setting',
                    'required' => false,
                    'desc' =>
                        '<div class="alert alert-info" role="alert">
                            <p class="alert-text">'. $this->l('You can now use the API Key for a more streamlined integration process. Please note that the traditional username and password authentication method will be deprecated in the near future. We strongly encourage you to switch to API Key authentication as soon as possible.') .'</p>
                            <p class="alert-text">'. $this->l('To obtain the API key please contact your sales manager or DPD:') .'</p>
                            <ul>
                                <li>'. $this->l('Estonia') .'<a href="sales@dpd.ee"> sales@dpd.ee</a></li>
                                <li>'. $this->l('Latvia') .'<a href="sales@dpd.lv"> sales@dpd.lv</a></li>
                                <li>'. $this->l('Lithuania') .'<a href="sales@dpd.lt"> sales@dpd.lt</a></li>
                            </ul>
                        </div>',
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Integration Country'),
                    'form_group_class' => 'dpd_setting',
                    'name' => 'dpd_countrysel',
                    'required' => false,
                    'options' => array(
                        'query' => array(
                            array(
                                'id'=> self::DPD_EE,
                                'name' => $this->l('Estonia'),
                            ),
                            array(
                                'id'=> self::DPD_LV,
                                'name' => $this->l('Latvia'),
                            ),
                            array(
                                'id'=> self::DPD_LT,
                                'name' => $this->l('Lithuania'),
                            ),
                        ),
                        'id' => 'id',
                        'name' => 'name'
                    )
                ),
                array('type' => 'hr', 'name'=> ''),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Name'),
                    'name' =>  'dpd_sender_name',
                    'form_group_class' => 'dpd_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Phone'),
                    'name' =>  'dpd_phone',
                    'form_group_class' => 'dpd_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Email'),
                    'name' =>  'dpd_email',
                    'form_group_class' => 'dpd_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - City'),
                    'name' =>  'dpd_city',
                    'form_group_class' => 'dpd_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Street'),
                    'name' =>  'dpd_street',
                    'form_group_class' => 'dpd_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Postal code'),
                    'name' =>  'dpd_zip',
                    'form_group_class' => 'dpd_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Return address - Country'),
                    'name' => 'dpd_country',
                    'lang' => false,
                    'form_group_class' => 'dpd_setting',
                    'options' => array(
                        'query' => $return_countries,
                        'id' => 'id',
                        'name' => 'name'
                    ),
                    'required' => false,
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
            ),
        );

        //LP express parcels sizes
        $parcelSizes = $this->getLpExpressParcelSizes();
        $fields_form[6]['form'] = array(
            'legend' => array(
                'title' => $this->l('LP Express carrier'),
            ),
            'input' => array(
                array(
                    'type' => 'switch',
                    'label' => $this->l('Enable LP Express carrier'),
                    'name' => 'lp_express_lt',
                    'form_group_class' => 'lp_express_lt_switch',
                    'required' => false,
                    'is_bool' => true,
                    'values' => array(
                        array(
                            'id' => 'lp_express_lt_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'lp_express_lt_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    )
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('LP Express WebServices Username'),
                    'name' =>  'lp_express_lt_username',
                    'form_group_class' => 'lp_express_lt_setting',
                    'hint' => $this->l('You get it from your Account Manager in LP Express'),
                    'required' => false,
                    'size' => 30,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Password'),
                    'name' =>  'lp_express_lt_password',
                    'form_group_class' => 'lp_express_lt_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('LP Express template'),
                    'name' =>  'lp_express_lt_template',
                    'form_group_class' => 'lp_express_lt_setting ' . (empty($parcelSizes) ? 'hidden' : ''),
                    'required' => false,
                    'options' => array(
                        'query' => $parcelSizes,
                        'id' => 'id',
                        'name' => 'name'
                    ),
                    'desc' => $this->l('All new orders will have this template as a default but can be individually changed in the order view. When changing the template size in the order view, the packing slip will also change.'),
                ),
                array('type' => 'hr', 'name'=> ''),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Name'),
                    'name' =>  'lp_express_lt_sender_name',
                    'form_group_class' => 'lp_express_lt_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Phone'),
                    'name' =>  'lp_express_lt_phone',
                    'form_group_class' => 'lp_express_lt_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Email'),
                    'name' =>  'lp_express_lt_email',
                    'form_group_class' => 'lp_express_lt_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Street'),
                    'name' =>  'lp_express_lt_street',
                    'form_group_class' => 'lp_express_lt_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Building'),
                    'name' =>  'lp_express_lt_building',
                    'form_group_class' => 'lp_express_lt_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - City'),
                    'name' =>  'lp_express_lt_city',
                    'form_group_class' => 'lp_express_lt_setting',
                    'required' => false,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Return address - Postal code'),
                    'name' =>  'lp_express_lt_zip',
                    'form_group_class' => 'lp_express_lt_setting',
                    'required' => false,
                ),
            ),
            'submit' => array(
                'title' => $this->l('Save'),
            ),
        );

        $helper->module = $this;
        $helper->name_controller = $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;

        $helper->default_form_language = $default_lang;
        $helper->allow_employee_form_lang = $default_lang;

        $helper->languages = Language::getLanguages(false);
        foreach ($helper->languages as $k => $language) {
            $helper->languages[$k]['is_default'] = (int)($language['id_lang'] == $helper->default_form_language);
        }

        $helper->title = $this->displayName;
        $helper->show_toolbar = true;
        $helper->toolbar_scroll = true;
        $helper->submit_action = 'submit'.$this->name;
        $helper->toolbar_btn = array(
            'save' => array(
                'desc' => $this->l('Save'),
                'href' => AdminController::$currentIndex.'&configure='.$this->name.'&save'.$this->name.
                '&token='.Tools::getAdminTokenLite('AdminModules'),
            ),
            'back' => array(
                'href' => AdminController::$currentIndex.'&token='.Tools::getAdminTokenLite('AdminModules'),
                'desc' => $this->l('Back to list')
            )
        );

        foreach ($this->getCarriers() as $carrier) {
            $this->fields[] = "carrier_" . $carrier["id_carrier"];
        }

        foreach ($this->fields as $param) {
            $field_values[$param] = Configuration::get(self::prefixed($param));
        }
        foreach ($this->lang_fields as $param)
        {
            foreach ($helper->languages as $key => $language)
            {
                $field_values[$param][$language['id_lang']] =
                    Configuration::get(self::prefixed($param), $language['id_lang']);
            }
        }

        $helper->fields_value = $field_values;

        $this->html .= $helper->generateForm($fields_form);
    }

    /**
     * Checks whether TMS is available for the shop
     * @return bool
     * @throws MKException
     */
    private function validateTmsAvailability()
    {
        if ($api = $this->getApi())
        {
            $params = array(
                'platform' => 'prestashop '._PS_VERSION_,
                'module' => 'MakeCommerce '.$this->version
            );
            $environment = array(
                'environment' => json_encode($params)
            );

            try {
                $shopConfig = $api->getShopConfig($environment);
            } catch (Exception $e) {
                PrestaShopLogger::addLog(
                    $e->getMessage(),
                    1,
                    null,
                    'MakeCommerce'
                );
                return false;
            }

            if (isset($shopConfig->features)) {
                foreach ($shopConfig->features as $value) {
                    if ($value->name === 'shipments_without_credentials') {
                        return $value->enabled;
                    }
                }
            }
        }

        return false;
    }

    /**
     * Gets available parcel sizes for LP Express
     * @return array
     */
    private function getLpExpressParcelSizes()
    {
        $options = array();
        if (class_exists('MakeCommerceLpExpress')) {
            $presets = MakeCommerceLpExpress::getParcelSizes();
            if (is_array($presets)) {
                foreach ($presets as $preset) {
                    $options[] = array('name' => $preset->label, 'id' => $preset->value);
                }
            }
        }

        return $options;
    }

    public function getPaymentMethodValues($country = null, $type = null)
    {
        $payment_methods = $this->getPaymentMethods();

        if (empty($payment_methods)) {
            return array();
        }

        $values = array();

        foreach ($payment_methods as $payment_method) {
            $method = $this->MkObjectToMethod($payment_method);

            if ((null === $type || $method->type == $type) &&
                (null === $country || $method->country == $country ||
                    ($method->country == self::COUNTRY_ALL && $type === null)
                )
            ) {
                $values[] = $method;
            }
        }
        $this->processMethods($values);

        return $values;
    }

    private function getMailVars()
    {
        return array(
            '{bankwire_owner}' => Configuration::get('BANK_WIRE_OWNER'),
            '{bankwire_details}' => nl2br(Configuration::get('BANK_WIRE_DETAILS')),
            '{bankwire_address}' => nl2br(Configuration::get('BANK_WIRE_ADDRESS'))
        );
    }

    public function confirmOrder($method_display_name = null)
    {
        $mailVars = $this->getMailVars();

        $this->validateOrder(
            $this->context->cart->id,
            Configuration::get(self::STATUS_NAME),
            $this->context->cart->getOrderTotal(),
            $this->displayName . ' ' . $method_display_name,
            NULL,
            $mailVars,
            $this->context->cart->id_currency,
            false,
            $this->context->customer->secure_key
        );
        $this->context->cookie->currentOrder = $this->currentOrder;
        $this->context->cookie->oldCartId = $this->context->cart->id;

        return new Order($this->currentOrder);
    }

    public static function prefixed($key)
    {
        return Tools::strtoupper(self::MODULE_NAME.'_'.$key);
    }

    public static function getConfig($key)
    {
        return Configuration::get(self::prefixed($key));
    }

    public function updateConfig($key, $value, $allow_html = false)
    {
        return Configuration::updateValue(self::prefixed($key), $value, $allow_html);
    }

    /**
     * Gets the api based on server configuration option (test/live)
     * @return Maksekeskus|null
     */
    public function getApi()
    {
        if (self::getConfig('server') == self::TEST) {
            $shop_id = self::getConfig('shop_id_test');
            $publishable_key = self::getConfig('publishable_key_test');
            $secret_key = self::getConfig('secret_key_test');
        } else {
            $shop_id = self::getConfig('shop_id');
            $publishable_key = self::getConfig('publishable_key');
            $secret_key = self::getConfig('secret_key');
        }

        if (!empty($shop_id) && !empty($publishable_key) && !empty($secret_key)) {
            $logger = new FileLogger(AbstractLogger::DEBUG);
            $logger->setFilename(_PS_ROOT_DIR_ . '/var/logs/' . self::LOG_DEBUG);
            self::$api = new Maksekeskus(
                $shop_id,
                $logger,
                $publishable_key,
                $secret_key,
                self::getConfig('server') == self::TEST
            );
        }

        return self::$api;
    }

    /**
     * Gets available payment methods for the shop, filtered by methods_order configuration
     * @param $cache
     * @return array|false|mixed
     */
    private function getPaymentMethods($cache = true)
    {
        $config_key = 'payment_methods';

        if ($cache) {
            $payment_methods_json = self::getConfig($config_key);
            if (!empty($payment_methods_json)) {
                $payment_methods_data = json_decode($payment_methods_json);
                if (
                    empty($payment_methods_data->updated) ||
                    $payment_methods_data->updated + self::CACHE_VALID_TIME < time()
                ) {
                    $payment_methods_data = null;
                }
            }
        }
        $payment_methods_data = null;
        if (!$cache || empty($payment_methods_data) ) {
            $api = $this->getApi();
            if (!is_null($api)) {
                $params = array(
                    'platform' => 'prestashop '._PS_VERSION_,
                    'module' => 'MakeCommerce '.$this->version
                );
                $environment = array(
                    'environment' => json_encode($params)
                );
                try {
                    $payment_methods_data = $api->getShopConfig($environment);
                    $payment_methods_data = $payment_methods_data->paymentMethods;
                } catch (Exception $e) {
                    PrestaShopLogger::addLog(
                        $e->getMessage(),
                        1,
                        null,
                        'MakeCommerce'
                    );
                    return false;
                }
            }

            $methods_order = self::getConfig('methods_order');

            $methods_order = array_filter(explode(',', $methods_order));
            $methods_order = array_map('trim', $methods_order);
            $methods_order = array_flip($methods_order);
            $payment_methods = array();
            $static_methods = array();
            if (isset($payment_methods_data) && $payment_methods_data) {
                foreach ($this->types as $type) {
                    foreach ($payment_methods_data->{$type} as $key => $payment_method) {
                        $payment_method->type = $type;
                        if (property_exists($payment_method, 'url')) {
                            unset($payment_method->url);
                        }

                        // change a couple of banklinks from country specific to all
                        if (in_array($payment_method->channel, ['N26 PIS', 'Revolut PIS'])
                            && !in_array($payment_method->channel, array_keys($static_methods))) {
                            $static_method = $payment_method;
                            $static_method->country = self::COUNTRY_ALL;
                            unset($static_method->countries);
                            $static_methods[$payment_method->channel] = $static_method;
                        }
                        // skip duplicate payment methods
                        if (in_array($payment_method->channel, array_keys($static_methods))) {
                            continue;
                        }

                        if (!property_exists($payment_method, 'country')) {
                            $payment_method->country = self::COUNTRY_ALL;
                        }

                        if (!empty($methods_order)) {
                            if(!array_key_exists($payment_method->name, $methods_order))
                                $payment_methods_default[] = $payment_method;
                        }
                        if (empty($methods_order)) {
                            $payment_methods[] = $payment_method;
                        } elseif (isset($methods_order[$payment_method->name])) {
                            $payment_methods[] = $payment_method;
                        }
                    }
                }
            }

            if (!empty($methods_order)){
                $payment_methods = json_decode(json_encode($payment_methods), true);
                usort($payment_methods, array($this,"paymentMethodOrder"));
                $payment_methods = json_decode(json_encode($payment_methods), false);
                $merged_payment_methods = array_merge($payment_methods, $payment_methods_default);
                $payment_methods = $merged_payment_methods;
            }

            if (isset($static_methods)) {
                foreach ($static_methods as $static) {
                    $payment_methods[] = $static;
                }
            }

            if (!empty($payment_methods)) {
                $object = new stdClass();
                $object->updated = time();
                $object->payment_methods = $payment_methods;
                $this->updateConfig($config_key, json_encode($object));
            }
        } else {
            $payment_methods = $payment_methods_data->payment_methods;
        }

        return $payment_methods;

    }

    public function paymentMethodOrder($a, $b){

        $methods_order = self::getConfig('methods_order');
        $methods_order = array_filter(explode(',', $methods_order));
        $methods_order = array_map('trim', $methods_order);

        $a = array_search($a["name"], $methods_order);
        $b = array_search($b["name"], $methods_order);

        if ($a === false && $b === false) {
            return 0;
        }
        else if ($a === false) {
            return 1;
        }
        else if ($b === false) {
            return -1;
        }
        else {
            return $a - $b;
        }
    }

    private function getCarriers() {
        return Carrier::getCarriers($this->context->employee->id_lang, true, false, false, null, 5);
    }

    private function createOrderState()
    {
        $orderStateExist = false;
        $orderStateId = Configuration::get(self::STATUS_NAME);
        $description = $this->l('Awaiting maksekeskus payment');

        if (strlen($description) > 64) {
            $description = substr($description, 0, 64);
        }

        if ($orderStateId) {
            $orderState = new OrderState($orderStateId);
            if($orderState->id && !$orderState->deleted) {
                $orderStateExist = true;
            }
        } else {
            $query = 'SELECT os.`id_order_state` '.
                'FROM `%1$sorder_state_lang` osl '.
                'LEFT JOIN `%1$sorder_state` os '.
                'ON osl.`id_order_state`=os.`id_order_state` '.
                'WHERE osl.`name`="%2$s" AND os.`deleted`=0';
            $orderStateId =  Db::getInstance()->getValue(sprintf($query, _DB_PREFIX_, $description));
            if ($orderStateId) {
                Configuration::updateValue(self::STATUS_NAME, $orderStateId);
                $orderStateExist = true;
            }
        }

        if (!$orderStateExist) {
            $languages = Language::getLanguages(false);
            $orderState = new OrderState();
            foreach ($languages as $lang) {
                $orderState->name[$lang['id_lang']] = $description;
            }

            $orderState->send_email = 0;
            $orderState->invoice = 0;
            $orderState->color = "lightblue";
            $orderState->unremovable = 0;
            $orderState->logable = 0;
            $orderState->delivery = 0;
            $orderState->hidden = 0;
            if($orderState->add()) {
                Configuration::updateValue(self::STATUS_NAME, $orderState->id);
                $orderStateExist = true;
            }
        }

        return $orderStateExist;
    }

    public function getImage($method)
    {
        $base_path = 'https://static.maksekeskus.ee/img/channel/lnd/';
        return  $base_path.$method.'.png';
    }

    public function getPaymentMethod($name, $payment_methods = array())
    {
        if (empty($name)) {
            return false;
        }

        if (empty($payment_methods)) {
            $payment_methods = $this->getPaymentMethodValues();
        }

        $result = false;
        foreach ($payment_methods as $method) {

            if ($method->code == $name) {
                $result = $method;
                break;
            }
        }

        return $result;
    }

    public function createPayment($transaction_id, $request_body){
        $api = $this->getApi();
        $api->createPayment($transaction_id, $request_body);
    }

    public function createTransaction(Method $method, $country)
    {
        // added because refreshing on card payment gave an error
        if (!empty($this->context->cookie->currentOrder)) {
            $order = new Order($this->context->cookie->currentOrder);
            // if order is already in paid status
            if ($order->getCurrentState() == Configuration::get('PS_OS_PAYMENT')) {
                $order = $this->confirmOrder($method->display_name);
            }
            // if cart id is different (meaning cart has changed) recreate the order
            elseif ($order->id_cart != $this->context->cart->id) {
                // set previous order state to cancelled
                $order->setCurrentState(6);
                $order->save();
                $order = $this->confirmOrder($method->display_name);
            }
            // change payment name if payment method was changed
            if ($order->payment !== $this->displayName . ' ' . $method->display_name) {
                $order->payment = $this->displayName . ' ' . $method->display_name;
                $order->save();
            }
        } else {
            $order = $this->confirmOrder($method->display_name);
        }
        $api = $this->getApi();
        $transaction = null;

        if (!is_null($api) && Validate::isLoadedObject($order)) {
            $currency = new Currency($order->id_currency);
            $customer = new Customer($order->id_customer);
            $address = new Address($order->id_address_delivery);
            $val_url = $this->context->link->getModuleLink($this->name, 'val');

            if($country == 'all')
                $country = Country::getIsoById($address->id_country);

            $data = array(
                'transaction' => array(
                    'amount' => number_format(round(Tools::ps_round($order->total_paid, 2),2),2,".",""),
                    'currency' => $currency->iso_code,
                    'reference' => $order->id,
                    'transaction_url' => array(
                        'return_url' => array(
                            'url' => $val_url,
                            'method' => 'POST'
                        ),
                        'cancel_url' => array(
                            'url' => $val_url,
                            'method' => 'POST'
                        ),
                        'notification_url' => array(
                            'url' => $val_url,
                            'method' => 'POST'
                        )
                    )
                ),
                'customer' => array(
                    'email' => $customer->email,
                    'ip' => $this->getCustomerIp(),
                    'country' => $country,
                    'locale' => Language::getIsoById($order->id_lang)
                )
            );

            $transaction = $api->createTransaction($data);
        }

        return $transaction;
    }

    public function getTransaction(string $transaction_id)
    {
        $api = $this->getApi();

        return $api->getTransaction($transaction_id);
    }

    public function getBankUrlFromTransaction($transaction, $method)
    {
        $url = false;

        if (!is_null($transaction)) {
            foreach ($transaction->payment_methods->{MakeCommerce::TYPE_BANK} as $payment_method) {
                if ($payment_method->name == $method) {
                    $url = $payment_method->url;
                    break;
                }
            }
            foreach ($transaction->payment_methods->{MakeCommerce::TYPE_OTHER} as $payment_method) {
                if ($payment_method->name == $method) {
                    $url = $payment_method->url;
                    break;
                }
            }
            foreach ($transaction->payment_methods->{MakeCommerce::TYPE_PAYLATER} as $payment_method) {
                if ($payment_method->name == $method) {
                    $url = $payment_method->url;
                    break;
                }
            }
        }

        return $url;
    }

    public function getJsDataFromTrasaction($transaction)
    {
        $order = new Order((int)$transaction->reference);
        $customer = new Customer($order->id_customer);

        if (self::getConfig('server') == self::TEST) {
            $js_src = 'https://payment.test.maksekeskus.ee/checkout/dist/checkout.js';
            $publishable_key = self::getConfig('publishable_key_test');
        } else {
            $js_src = 'https://payment.maksekeskus.ee/checkout/dist/checkout.js';
            $publishable_key = self::getConfig('publishable_key');
        }

        $selectedMethod = Tools::getValue('method');
        $preselect = 'CC';
        if (stristr($selectedMethod, 'apple_pay')) {
            $preselect = 'AP';
        } elseif (stristr($selectedMethod, 'google_pay')) {
            $preselect = 'GP';
        }

        return array(
            'prefill' => self::getConfig('prefill_form'),
            'js_src' => $js_src,
            'publishable_key' => $publishable_key,
            'transaction_id' => $transaction->id,
            'currency' => $transaction->currency,
            'amount' => number_format(round($transaction->amount, 2),2,".",""), // number_format(round(Tools::ps_round($cart->getOrderTotal(true, 1), 2),2),2),
            'customer_email' => $transaction->customer->email,
            'customer_name' => $customer->firstname.' '.$customer->lastname,
            'shop_name' => $this->context->shop->name,
            'description' => sprintf($this->l('Order #%d'), $transaction->reference),
            'locale' => Language::getIsoById($order->id_lang),
            'quick_mode' => true,
            'completed' => 'makecommerce_cc_complete', // completed function name for CC payments
            'pre_select' => $preselect
        );
    }

    public function getCountries($payment_methods = null)
    {
        if (null === $payment_methods) {
            $payment_methods = $this->getPaymentMethods();
        }

        $countries = array();
        if(isset($payment_methods) && $payment_methods){
            foreach ($payment_methods as $payment_method) {
                if (!isset($countries[$payment_method->country])) {
                    $countries[$payment_method->country] = 1;
                }
            }
        }
        return array_keys($countries);
    }

    public function appendPaymentMethods(&$types, $with_credit_cards = false)
    {
        $payment_methods = $this->getPaymentMethods();
        $grouped_methods = array();
        $cards = array();
        if (isset($payment_methods) && $payment_methods) {
            foreach ($payment_methods as $payment_method) {
                $method = $this->MkObjectToMethod($payment_method);
                $key = sprintf('%s_%s', $method->type, $method->country);
                if (!isset($grouped_methods[$key])) {
                    $grouped_methods[$key] = array();
                }
                $grouped_methods[$key][] = $method;

                if ($with_credit_cards && $method->type == self::TYPE_CARD) {
                    $cards[] = $method;
                }
            }
        }
        foreach ($types as &$type) {
            $key = $type->getKey();
            if (isset($grouped_methods[$key])) {
                $type->methods = array_merge($grouped_methods[$key], $cards);
                $this->processMethods($type->methods, self::IMG_SIZE_S);
            }
        }
    }

    public function hookDisplayOrderDetail()
    {
        if (empty($id_order = Tools::getValue('id_order'))) {
            return false;
        }
        $order = new Order($id_order);
        $trackingLink = self::getTrackingUrl($order);

        // if tracking link is not returned dont display anything
        if (!$trackingLink) {
            return false;
        }

        $this->smarty->assign(array(
            'tracking_link' => $trackingLink
        ));
        return $this->display(__FILE__, 'orderDetail.tpl');
    }

    public function hookHeader()
    {
        $this->context->controller->addJS($this->getPathUri().'views/js/makecommerce.js');
        $this->context->controller->addCSS($this->getPathUri().'views/css/makecommerce.css');
    }

    public function hookActionOrderGridDefinitionModifier(array $params)
    {
        $params['definition']->getBulkActions()->add(
                (new SubmitBulkAction('parcellabels'))
                    ->setName($this->l('Print parcel labels')) // l s='Print parcel labels'
                    ->setOptions([
                        // in most cases submit action should be implemented by module
                        'submit_route' => 'admin_makecommerce_orders_bulk_printlabels',
                    ])
            );
    }

    public function hookBackOfficeHeader()
    {
        $this->addCarriersFields();
        $this->context->controller->addJquery();
        $this->context->controller->addJS($this->getPathUri().'views/js/makecommerce.js');
        $this->context->controller->addJS($this->getPathUri().'views/js/carrier_script.js');

        if (Module::isInstalled('makecommerceomniva') || Module::isInstalled('makecommercesmartpost') || Module::isInstalled('makecommercedpd')){
            $this->smarty->assign(array(
                'parcel_lable' => $this->context->link->getModuleLink($this->name, 'parcellabels'),
                'loader_img' => 'https://'.Tools::getMediaServer(_PS_IMG_)._PS_IMG_.'loader.gif'
            ));
            return $this->display(__FILE__, 'bulkaction.tpl');
        }
    }

    public function hookDisplayBackOfficeHeader()
    {
        return $this->hookBackOfficeHeader();
    }

    public  function hookDisplayAdminOrder($params)
    {
        return $this->hookDisplayAdminOrderMain($params);
    }

   public function hookDisplayHeader()
   {
       $this->hookHeader();
   }

    public function hookDisplayPaymentReturn($params)
    {
        $order = $params['order'];
        $trackingUrl = $this->getTrackingUrl($order);

        if ($order->hasBeenPaid())
        {
            $status = 'ok';
        }
        else
        {
            $status = 'error';
        }

        $this->smarty->assign(array(
            'status' => $status,
            'tracking_link' => $trackingUrl,
            'link_to_order' => $this->getOrderConfUrl($order)
        ));
        return $this->display(__FILE__, 'payment_return.tpl');
    }

    public function hookPaymentOptions($params) {
        $methods_display = (int)self::getConfig('methods_display');
        $credit_cards_separately = (bool)self::getConfig('credit_display');
        $methods_named = (bool)self::getConfig('methods_named');
        $countries = $this->getCountries();
        $invoice_address = new Address((int)$this->context->cart->id_address_delivery);
        $current_country = Tools::strtolower(Country::getIsoById((int)$invoice_address->id_country));

        switch ($methods_display) {
            case self::DISPLAY_ORDER_PAGE:
                $payment_methods = $this->getPaymentMethodValues();
                if(count($countries)> 1){
                    $show_country_code = TRUE;
                }else{
                    $show_country_code = FALSE;
                }
                $this->context->smarty->assign(array(
                    'show_country_code' => $show_country_code,
                    'methods_named' => $methods_named
                ));
                break;

            case self::DISPLAY_WIDGET:
                $payment_methods = $this->getPaymentMethodValues();
                if(count($countries)> 1){
                    $show_country_code = TRUE;
                }else{
                    $show_country_code = FALSE;
                }
                $this->appendPaymentMethods($payment_methods, !$credit_cards_separately);
                $this->context->smarty->assign(array(
                    'widget_title' => trim(Configuration::get(
                        self::prefixed('widget_title'),
                        $this->context->language->id)),
                    'widget' => TRUE,
                    'show_country_code' => $show_country_code,
                    'separate_group' => $credit_cards_separately,
                    'methods_named' => $methods_named
                ));
                break;
        }

        if (empty($payment_methods)) {
            return;
        }
        $img_size = self::getConfig('logo_size');
        $this->processMethods($payment_methods, $img_size);
        $this->context->smarty->assign(array(
            'payment_methods' => $payment_methods,
            'logo_size' => $img_size,
            'mk_ajax_url' => $this->context->link->getModuleLink($this->name, 'ajax'),
            'order_total' => $this->context->cart->getOrderTotal(),
            'flag_url' => _MODULE_DIR_,
            'countries' => $countries,
            'current_country' => $current_country,
            'one_page_checkout' => (bool)Module::isEnabled('thecheckout')
        ));

        $newOption = new PaymentOption();
        $newOption->setModuleName($this->name)
            ->setCallToActionText($this->l('Pay by MakeCommerce'))
            ->setAction($this->context->link->getModuleLink($this->name, 'confirmation', array(), true))
            ->setAdditionalInformation($this->fetch('module:makecommerce/views/templates/hook/payments.tpl'));
        $payment_options = [
            $newOption,
        ];

        return $payment_options;
    }

    public function hookDisplayAdminOrderMain($params)
    {
        $error = '';
        $id_order = $params['id_order'];
        $order = new Order((int)$id_order);
        $maksekeskusPayment = FALSE;
        foreach ($order->getOrderPaymentCollection() as $payment) {
            if(stristr($payment->payment_method, 'MakeCommerce')) {
                $maksekeskusPayment = TRUE;
                break;
            }
        }
        if ($maksekeskusPayment) {

            if (Tools::isSubmit('submitMKRefund')) {
                $this->_doTotalRefund($id_order);
            }

            if (Tools::isSubmit('submitMKRefundPartial')) {
                $refund_amount = Tools::getValue('mk_partial_refund');
                $order_total = $order->total_paid;
                $refunded_total = $this->refundedAmount($id_order);
                $remaining_amount = $order_total - $refunded_total;

                if (isset($refund_amount) && $refund_amount > 0) {
                    if ($remaining_amount - $refund_amount >= 0) {
                        $this->_doPartialRefund($id_order, $refund_amount);
                    } else {
                        $error = $this->l('Refund amount greater than available to refund.');
                    }
                } else {
                    $error = $this->l('Invalid refund value.');
                }
            }

            if ($this->context->cookie->__isset('refund_status')) {
                // if ($this->context->cookie->__get('refund_status') != "")
                    $error = $this->context->cookie->__get('refund_status');
                $this->context->cookie->__unset('refund_status');
                $this->context->cookie->write();
            }
            $this->smarty->assign(array(
                'error' => $error,
                'id_order' => $id_order,
                'refunded' => $this->refundedAmount($id_order),
                'refund_details' => $this->getRefundDetails($id_order),
                'total_amount' => $order->total_paid
            ));
            return $this->display(_PS_MODULE_DIR_ .'makecommerce/makecommerce.php', 'adminOrder.tpl');
        }
    }

    protected function _doTotalRefund($id_order)
    {
        $order = new Order($id_order);
        $amount = number_format(round(Tools::ps_round($order->total_paid, 2),2),2,".","");
        $transaction_id = Db::getInstance()->getValue(
            'SELECT `transaction_id`
            FROM `'._DB_PREFIX_.'order_payment`
            WHERE order_reference = \''.$order->reference.'\''
        );

        $request_body = array(
            'amount' => $amount,
            'comment' => 'Order: '.$id_order.' refund'
        );

        $api = $this->getApi();
        try {
            $api->createRefund($transaction_id, $request_body);
            $this->saveRefundDetails($id_order, $amount);
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                    $e->getMessage(),
                    1,
                    null,
                    'MakeCommerce'
            );
            $this->context->controller->errors[] = $e->getMessage();

            if ($e instanceof MKException) {
                $error_raw = json_decode($e->getRawContent(), true);
                $error_msg = $error_raw['message'];
            } else {
                $error_msg = $this->l('Total refund failed. See log for details.');
            }

            $this->context->cookie->__set('refund_status', $error_msg);
            $this->context->cookie->write();
            Tools::redirect($_SERVER['HTTP_REFERER']);
        }
        $order->setCurrentState(Configuration::get('PS_OS_REFUND'));
        Tools::redirect($_SERVER['HTTP_REFERER']);
    }

    protected function _doPartialRefund($id_order, $amount)
    {
        $order = new Order($id_order);
        $partial_amount = number_format(round(Tools::ps_round($amount, 2),2),2,".","");
        $transaction_id = Db::getInstance()->getValue(
            'SELECT `transaction_id`
            FROM `'._DB_PREFIX_.'order_payment`
            WHERE order_reference = \''.$order->reference.'\''
        );

        $request_body = array(
            'amount' => $partial_amount,
            'comment' => 'Order: '.$id_order.' refund'
        );
        $api = $this->getApi();
        try {
            $api->createRefund($transaction_id, $request_body);
            $this->saveRefundDetails($id_order, $partial_amount);
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                    $e->getMessage(),
                    1,
                    null,
                    'MakeCommerce'
            );
            if ($e instanceof MKException) {
                $error_raw = json_decode($e->getRawContent(), true);
                $error_msg = $error_raw['message'];
            } else {
                $error_msg = $this->l('Total refund failed. See log for details.');
            }
            $this->context->controller->errors[] = $e->getMessage();
            $this->context->cookie->__set('refund_status', $error_msg);
            $this->context->cookie->write();
            Tools::redirect($_SERVER['HTTP_REFERER']);
        }
        $order->setCurrentState(Configuration::get('PS_OS_REFUND'));
        Tools::redirect($_SERVER['HTTP_REFERER']);
    }

    private function refundedAmount($id_order)
    {
        $refunded_total = Db::getInstance()->ExecuteS(
            'SELECT SUM(refund_amount) AS total FROM `'._DB_PREFIX_.'makecommerce_refunds` WHERE id_order = \''.$id_order.'\''
        );
        return $refunded_total[0]['total'];
    }

    private function saveRefundDetails($id_order, $amount)
    {
        Db::getInstance()->Execute(
            'INSERT INTO `'._DB_PREFIX_.'makecommerce_refunds` (`id_order`, `refund_amount`, `refund_date`)
                    VALUES('.(int)$id_order.', \''.$amount.'\', \''.date('Y-m-d H:i:s').'\')'
        );
    }

    private function getRefundDetails($id_order)
    {
        return Db::getInstance()->ExecuteS(
            'SELECT * FROM `'._DB_PREFIX_.'makecommerce_refunds` WHERE id_order = \''.$id_order.'\''
        );
    }

    protected function processMethods(array &$methods)
    {

        foreach ($methods as &$method)
        {
            $params = array(
                'method' => $method->code,
                'country' => $method->country
            );

            $method->link = $this->context->link->getModuleLink(
                $this->name,
                (empty($method->methods) ? 'confirmation' : 'payments'),
                $params
            );
            $method->img = $method->logo_url;
        }
    }

    public function MkObjectToMethod(stdClass $mk_method)
    {
        if (!(isset($mk_method->min_amount) && $mk_method->min_amount > 0))
            $mk_method->min_amount = 0;
        if (!(isset($mk_method->max_amount) && $mk_method->max_amount > 0))
            $mk_method->max_amount = 0;
        return new Method(
            $mk_method->name,
            $mk_method->display_name,
            $mk_method->display_name,
            $mk_method->country,
            $mk_method->type,
            $mk_method->logo_url,
            $mk_method->min_amount,
            $mk_method->max_amount
        );
    }

    public function getCustomerIp()
    {
        if (
            !empty($_SERVER['HTTP_CF_CONNECTING_IP']) &&
            filter_var($_SERVER['HTTP_CF_CONNECTING_IP'], FILTER_VALIDATE_IP)
        ) {
            return $_SERVER['HTTP_CF_CONNECTING_IP'];
        }

        return $_SERVER['REMOTE_ADDR'];
    }

    /**
     * Returns a tracking url for a carrier
     * @param Order $order
     * @return false|string
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public static function getTrackingUrl(Order $order)
    {
        $countries = array(
            'estonia' => 'ee',
            'latvia' => 'lv',
            'lithuania' => 'lt',
            'finland' => 'fi'
        );

        $shipping = $order->getShipping();

        // get tracking number
        $trackingNumber = $shipping[0]['tracking_number'];
        if (empty($trackingNumber)) {
            return false;
        }

        // get carrier name needs to be passed as omniva, smartpost, dpd etc
        $carrierName = explode(' ', $shipping[0]['carrier_name'])[0];
        $carrierName = (strtolower($carrierName) === 'lp') ? 'lpexpress' : $carrierName;

        if  ($order instanceof Order) {
            $lang = new Language($order->id_lang);
            $address = new Address($order->id_address_delivery);
            // get destination country (Estonia, Latvia, etc)
            $country = $address->country;
        }

        $module_mode = self::getConfig('server');
        return MakeCommerceCarrierModule::TRACKING_URLS[$module_mode] .
            $trackingNumber .
            '?carrier=' .
            strtolower($carrierName) .
            '&dst=' . $countries[strtolower($country)] .
            '&lang=' . $lang->iso_code
        ;
    }

    protected function clearPaymentCache()
    {
        $query = 'DELETE FROM %sconfiguration WHERE `name` LIKE \'%s%%\'';
        Db::getInstance()->execute(sprintf(
            $query,
            _DB_PREFIX_,
            self::prefixed('payment_methods')
        ));
    }

    protected function clearCarriersCache()
    {
        $cacheCleared = false;
        try {
            $tableExists = Db::getInstance()->executeS('SELECT `carriers_list` FROM `'._DB_PREFIX_.'makecommerce_carriers`');
            if ($tableExists) {
                // delete carriers cache upon update
                $cacheCleared = Db::getInstance()->delete('makecommerce_carriers');
            }
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                $e->getMessage(),
                3,
                null,
                'MakeCommerce'
            );
        }

        return $cacheCleared;
    }

    public function getOrderConfUrl($order)
    {
        return $this->context->link->getPageLink(
            'order-confirmation',
            true,
            $order->id_lang
        );
    }

    private function addCarriersFields()
  {
        $carriers = Carrier::getCarriers($this->context->employee->id_lang, true, false, false, null, 5);
        foreach ($carriers as $carrier) {
            $this->fields[] = 'carrier_'.$carrier['id_carrier'];
        }
    }

    /**
     * Checks whether product is in stock in sufficient amount and if out of stock product ordering is disabled
     * @param $id_cart
     * @return bool
     */
    public function validateCart($id_cart)
    {
        $valid = true;
        $cart = new Cart($id_cart);
        $products = $cart->getProducts();

        foreach ($products as $product) {
            if ($product['cart_quantity'] > $product['quantity_available'] && $product['out_of_stock'] != 1 && $product['allow_oosp'] != 1) {
                $valid = false;
            }
        }

        return $valid;
    }

    public function hookActionFrontControllerSetMedia($params)
    {
        if (Tools::getValue('submitReorder')) {
            self::restoreCart();
        }
    }

    public static function restoreCart()
    {
        $context = Context::getContext();
        $oldCartId = $context->cookie->oldCartId;
        if (!$context->cart->id && $oldCartId) {
            $oldCart = new Cart($oldCartId);
            $duplication = $oldCart->duplicate();

            if ($duplication && $duplication['success']) {
                $context->cookie->id_cart = $duplication['cart']->id;
                $context->cart = $duplication['cart'];
                CartRule::autoAddToCart($context);
            }

            unset($context->cookie->oldCartId);
        }
    }

}
