<?php

class MakeCommerceCarrierModule extends CarrierModule
{
    const MODULES_COUNT = 'makecommerce_modules_count';

    const CACHE_VALID_TIME = 3600;

    const VALID_COUNTRIES = array();

    const CARRIER_METHODS = array();

    const TRACKING_URLS = array(
        0 => 'https://tracking.test.makecommerce.net/',
        1 => 'https://tracking.makecommerce.net/'
    );

    static $jsIsIncluded = false;

    public $logo_url;

    public function install()
    {

        if (!parent::install() OR
            !$this->registerHook('extraCarrier') OR
            !$this->registerHook('beforeCarrier') OR
            !$this->registerHook('actionValidateOrder') OR
            !$this->registerHook('actionPaymentConfirmation') OR
            !$this->registerHook('actionObjectAddressUpdateAfter') OR
            !$this->registerHook('validateCarriers') OR
            !$this->registerHook('displayHeader') OR
            !$this->registerHook('displayAdminOrder') OR
            !$this->registerHook('displayCarrierExtraContent') OR
            !$this->registerHook('displayAdminOrderSide') OR
            !$this->installCarrier() OR
            !$this->createCarrierTable() OR
            !$this->createLPexpressTable() OR
            !$this->saveTerminals()
        ) {
            return false;
        }

        return true;
    }

    public function createCarrierTable()
    {

        if (!Db::getInstance()->execute('
			CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'makecommerce_carriers` (
			  `carrier` varchar(255) NOT NULL,
			  `carriers_list` mediumtext NOT NULL
			) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=UTF8;
		')) {
            return false;
        }
        return true;
    }

    public function createLPexpressTable()
    {
        $sql = 'CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'makecommerce_lpexpress` (
			  `shipmentid` varchar(64) PRIMARY KEY,
			  `trackingNumber` varchar(64),
			  `lpExpressCartIdentifier` varchar(64),
			  `parcelSize` varchar(64)
			) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=UTF8;';

        if (!Db::getInstance()->execute($sql)) {
            return false;
        }

        return true;
    }

    public function disable($force_all = false)
    {
        if ($courier_carrier_id = $this->getCarrierId(true,'courier')) {
            $courier_carrier = new Carrier($courier_carrier_id);
            $courier_carrier->active = false;
            $courier_carrier->save();
        }
        if ($parcel_carrier_id = $this->getCarrierId(true)) {
            $parcel_carrier = new Carrier($parcel_carrier_id);
            $parcel_carrier->active = false;
            $parcel_carrier->save();
        }

        return parent::disable($force_all); // TODO: Change the autogenerated stub
    }

    public function enable($force_all = false)
    {
        if ($courier_carrier_id = $this->getCarrierId(false,'courier')) {
            $courier_carrier = new Carrier($courier_carrier_id);
            $courier_carrier->active = true;
            $courier_carrier->save();
        }
        if ($parcel_carrier_id = $this->getCarrierId(false)) {
            $parcel_carrier = new Carrier($parcel_carrier_id);
            $parcel_carrier->active = true;
            $parcel_carrier->save();
        }

        return parent::enable($force_all); // TODO: Change the autogenerated stub
    }

    public function uninstall()
    {
        Db::getInstance()->update(
            'carrier',
            array('deleted' => 1),
            sprintf('`external_module_name` = \'%s\'', $this->name)
        );

        Configuration::updateValue(Tools::strtoupper('makecommerce_'.$this->carrier_name), 0);

        $this->removeModule();

        if (!parent::uninstall() OR
            !$this->unregisterHook('beforeCarrier') OR
            !$this->unregisterHook('actionValidateOrder') OR
            !$this->unregisterHook('actionPaymentConfirmation') OR
            !$this->unregisterHook('validateCarriers') OR
            !$this->unregisterHook('displayHeader') OR
            !$this->unregisterHook('actionObjectAddressUpdateAfter') OR
            !$this->unregisterHook('displayCarrierExtraContent') OR
            !$this->unregisterHook('displayAdminOrder') OR
            !$this->unregisterHook('displayAdminOrderSide')
        ) {
            return false;
        }

        //enable carrier in makecommerce module config
        $carrierPrefix = $this->getCarrierPrefix();
        Configuration::updateValue(Tools::strtoupper('makecommerce_'.$carrierPrefix), 0);

        return true;
    }

    public function installCarrier()
    {
        foreach ($this::CARRIER_METHODS as $carrierMethod) {
            $carrier = new Carrier();
            $carrier->name = $this->carrier_front_name . ' ' .str_replace('_', ' ', $carrierMethod);
            $carrier->id_tax_rules_group = 0;
            $carrier->id_zone = 1;
            $carrier->delay = array();
            $carrier->range_behavior = 1;
            $carrier->is_module = true;
            $carrier->shipping_external = true;
            $carrier->external_module_name = $this->name;
            $carrier->need_range = true;
            $carrier->max_width = '38';
            $carrier->max_height = '41';
            $carrier->max_depth = '64';
            $carrier->max_weight = '30';

            $languages = Language::getLanguages(true);
            foreach ($languages as $language) {
                $carrier->delay[(int)$language['id_lang']] = '2-3 days';
            }

            if ($carrier->add()) {
                $groups = Group::getGroups(true);
                foreach ($groups as $group) {
                    Db::getInstance()->insert(
                        'carrier_group',
                        array(
                            'id_carrier' => (int)($carrier->id),
                            'id_group' => (int)($group['id_group'])
                        )
                    );
                }

                $this->addZonesToCarrier($carrier);
                $this->addRangeToCarrier($carrier);

                //enable carrier in makecommerce module config
                $carrierPrefix = $this->getCarrierPrefix();
                Configuration::updateValue(Tools::strtoupper('makecommerce_'.$carrierPrefix), 1);

                if ($carrierMethod === 'courier') {
                    Configuration::updateValue(Tools::strtoupper('makecommerce_courier_'.$carrierPrefix), $carrier->id);
                }

                // carrier logo path
                $carrierLogoPath = _PS_SHIP_IMG_DIR_.'/'.(int)$carrier->id.'.jpg';
                // check if file exists
                if (!$this->downloadCarrierLogo($this->logo_url, $carrierLogoPath)) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * Add delivery zones to each carrier
     * Creates a new Zone for each country defined in CarrierModule "VALID_COUNTRIES" and add it do carrier zones
     */
    private function addZonesToCarrier(Carrier $carrier)
    {
        $id_lang = Language::getIdByIso('en');
        $zones = Zone::getZones(true);

        // create zones for each country and add valid zones to carriers
        foreach ($this::VALID_COUNTRIES as $iso_code) {
            $id_country = (int) Country::getByIso($iso_code);
            $country = new Country($id_country);
            $zonesAdded = false;

            foreach ($zones as $zone) {
                if (mb_strtolower($zone['name']) === mb_strtolower($country->name[$id_lang])) {
                    $carrier->addZone($zone['id_zone']);
                    $zonesAdded = true;
                }
            }

            if (!$zonesAdded) {
                $newZone = new Zone();
                $newZone->name = $country->name[$id_lang];
                $newZone->active = true;
                $newZone->add();

                // add newly created zone to country
                $country->id_zone = $newZone->id;
                $country->save();
                // add newly created zone to carrier
                $carrier->addZone($newZone->id);
            }
        }

    }

    /**
     * Add ranges to newly created Zones
     */
    private function addRangeToCarrier(Carrier $carrier)
    {
        $rangePrice = new RangePrice();
        $rangePrice->id_carrier = $carrier->id;
        $rangePrice->delimiter1 = '0';
        $rangePrice->delimiter2 = '9999999';
        $rangePrice->add();

        $rangeWeight = new RangeWeight();
        $rangeWeight->id_carrier = $carrier->id;
        $rangeWeight->delimiter1 = '0';
        $rangeWeight->delimiter2 = '30';
        $rangeWeight->add();

        // get active zones for a carrier
        $zones = $carrier->getZones();
        foreach ($zones as $zone) {
            $price_list[] = [
                'id_carrier' => (int)($carrier->id),
                'id_range_price' => (int)($rangePrice->id),
                'id_range_weight' => NULL,
                'id_zone' => (int)($zone['id_zone']),
                'price' => '0'
            ];

            $carrier->addDeliveryPrice($price_list);

            $price_list2[] = array(
                'id_carrier' => (int)($carrier->id),
                'id_range_price' => NULL,
                'id_range_weight' => (int)($rangeWeight->id),
                'id_zone' => (int)($zone['id_zone']),
                'price' => '0'
            );
            $carrier->addDeliveryPrice($price_list2);
        }
    }

    private function downloadCarrierLogo($url, $pathToDownload)
    {
        if (empty($url)) {
            return false;
        }
        // if image already exist delete it.
        if (file_exists($pathToDownload)) {
            unlink($pathToDownload);
        }

        $fp = fopen($pathToDownload, "w");

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_FILE, $fp);
        curl_setopt($ch, CURLOPT_HEADER, 0);
        $result = curl_exec($ch);

        curl_close($ch);
        fclose($fp);

        return $result;
    }

    public function hookActionPaymentConfirmation($params)
    {
        $order = new Order($params['id_order']);
        if ($this->getCookie($order->id_address_delivery)) {
            unset($this->context->cookie->{$this->name . '_' . $order->id_address_delivery});
        }
        $carrier = new Carrier($order->id_carrier);
        if($carrier->external_module_name == $this->name){
            $this->createShipments($order);
        }
    }

    public function hookValidateCarriers($params)
    {
        if(isset($params['cart']) && Validate::isLoadedObject($params['cart'])) {
            $carrier = new Carrier($params['cart']->id_carrier);
            $extracarrier_data = $this->getCookie($params['cart']->id_address_delivery);
            if ($carrier->is_module && $carrier->external_module_name == $this->name &&
                (!isset($extracarrier_data['terminal_id'] ) ||
                    $extracarrier_data['terminal_id'] == 0)
            ) {
                return '<p class="warning">'.Tools::displayError('Error: Please choose a parcel terminal.').'</p>';
            }
        }
        return false;
    }

    /**
     * Saves selected terminal info to address object
     * @param $carrier_name - name of the carrier
     * @param $terminal_id - selected terminal id
     * @param $id_address_delivery - delivery address id
     * @return Address
     */
    public static function addTerminalAddress($carrier_name, $terminal_id, $id_address_delivery)
    {
        $terminals_json = Db::getInstance()->getValue(
            'SELECT `carriers_list` FROM `'._DB_PREFIX_.'makecommerce_carriers` WHERE `carrier` = \''. $carrier_name.'\''
        );

        $terminal_data = json_decode($terminals_json);
        $terminals = json_decode(json_encode($terminal_data->terminals), true);

        foreach($terminals as $terminal){
            if(isset($terminal['id']) && $terminal['id'] == $terminal_id){
                $order_terminal = new stdClass();
                $order_terminal->id = $terminal['id'];
                $order_terminal->name = $terminal['name'];
                if($terminal['country'] == 'EE'){
                    $order_terminal->address = $terminal['address'];
                }else{
                    $order_terminal->address = $terminal['city'];
                }
                $order_terminal->city = $terminal['city'];
                break;
            }
        }

        $address = new Address($id_address_delivery);
        $terminal_address = new Address();
        $terminal_address->id_country = $address->id_country;
        $terminal_address->id_state = $address->id_state;
        $terminal_address->id_customer = $address->id_customer;
        $terminal_address->id_manufacturer = $address->id_manufacturer;
        $terminal_address->id_supplier = $address->id_supplier;
        $terminal_address->id_warehouse = $address->id_warehouse;
        $terminal_address->alias = 'Terminal address';
        $terminal_address->company = $address->company;
        $terminal_address->lastname = $address->lastname;
        $terminal_address->firstname = $address->firstname;
        if (isset($order_terminal)) {
            $terminal_address->address1 = $order_terminal->name;
            $terminal_address->address2 = $order_terminal->address;
            $terminal_address->other = $order_terminal->id;
            $terminal_address->city = $order_terminal->city;
        } else {
            $terminal_address->address1 = $address->address1;
            $terminal_address->address2 = $address->address2;
            $terminal_address->other = $address->other;
            $terminal_address->city = $address->city;
        }
        $terminal_address->postcode = $address->postcode;
        $terminal_address->phone = $address->phone;
        $terminal_address->phone_mobile = $address->phone_mobile;
        $terminal_address->vat_number = $address->vat_number;
        $terminal_address->dni = $address->dni;
        $terminal_address->deleted = 1;
        $terminal_address->add();

        return $terminal_address;
    }

    public function hookActionValidateOrder($params)
    {
        if($params['order']->id_carrier == $this->getCarrierId()) {
            $extracarrier_data = $this->getCookie($params['order']->id_address_delivery);

            $terminal_address = self::addTerminalAddress($this->carrier_name, $extracarrier_data['terminal_id'], $params['order']->id_address_delivery);

            $order = new Order($params['order']->id);
            $order->id_address_delivery = $terminal_address->id;
            $params['order']->id_address_delivery = $terminal_address->id;
            $order->save();

        }
    }

    public function getOrderShippingCost($cart, $shipping_cost)
    {
        return $shipping_cost;
    }

    public function getOrderShippingCostExternal($cartObject)
    {
        return $this->getOrderShippingCost($cartObject, 0);
    }

    public function getFileName()
    {
        return $this->getLocalPath().$this->name.'.php';
    }

    public function prefixed($key)
    {
        return Tools::strtoupper($this->name.'_'.$key);
    }

    public function getConfig($key)
    {
        return Configuration::get($this->prefixed($key));
    }

    public function updateConfig($key, $value, $allow_html = false)
    {
        return Configuration::updateValue($this->prefixed($key), $value, $allow_html);
    }

    public function removeModule()
    {
        $modules_serialized = Configuration::getGlobalValue(self::MODULES_COUNT);
        $modules = Tools::unSerialize($modules_serialized);
        if (!is_array($modules)) {
            $modules = array();
        }
        if (isset($modules[$this->name])) {
            unset($modules[$this->name]);
        }
        Configuration::updateGlobalValue(self::MODULES_COUNT, serialize($modules));
    }

    public function hookDisplayAdminOrder($params)
    {
        return $this->hookDisplayAdminOrderSide($params);
    }


    public function hookDisplayAdminOrderSide($params)
    {
        $order = new Order($params['id_order']);
        $output = '';
        $makecommerceCarrier = FALSE;

        foreach ($order->getShipping() as $carrier){
            $carrire_obj = new Carrier($carrier['id_carrier']);
            if($carrire_obj->external_module_name == $this->name){
                $makecommerceCarrier = TRUE;
                break;
            }
        }

        if($makecommerceCarrier){
            // prints label for order carrier
            if (Tools::isSubmit('submitMKLabel')) {
                $label_url = $this->getParcelLabel($order);
                $this->smarty->assign(array(
                    'label_url' => $label_url,
                ));
            }

            $shipment_id = $this->getShipmentId($order);
            // registers order carrier
            if (Tools::isSubmit('submitMKRegister')) {
                if(empty($shipment_id)){
                    $error = $this->createShipments($order);
                    $shipment_id = $this->getShipmentId($order);

                    if (!empty($error) && strstr($error, "Terminal does not support the chosen shipment's size")) {
                        $output .= $this->displayError($this->l("Terminal does not support the chosen shipment's size. Please choose another!"));
                        $error = null;
                    }
                    elseif (!empty($shipment_id)) {
                        $output .= $this->displayConfirmation('New shipment registered #' . $shipment_id);
                    }

                    $label_url = $this->getParcelLabel($order);
                    $this->smarty->assign(array(
                        'label_url' => $label_url,
                        'error' => $error,
                    ));
                }
            }

            // handle shipping aadress update request
            if (Tools::isSubmit('submitMKParcelDestinationUpdate')) {
                $terminal_address = self::addTerminalAddress(Tools::getValue('carrier_name'), Tools::getValue('terminals'), $order->id_address_delivery);
                $order->id_address_delivery = $terminal_address->id;
                $order->save();
                // register order again with updated terminal data
                $this->createShipments($order);
            }

            // LP express specific action
            if (Tools::isSubmit('submitMKParcelChange')) {
                $defaultParcelSize = Configuration::get('MAKECOMMERCE_LP_EXPRESS_LT_TEMPLATE');
                // prevent duplicate submits

                if (!empty($shipment_id) && $this->getLpExpressParcelSize($shipment_id) != Tools::getValue('parcelSize') ||
                    (!$shipment_id && $defaultParcelSize != Tools::getValue('parcelSize'))
                ) {
                    $error = $this->createShipments($order);
                    $new_shipment_id = $this->getShipmentId($order);

                    if ($shipment_id != $new_shipment_id) {
                        $output .= $this->displayConfirmation('New shipment registered #' . $shipment_id);
                    }

                    if (strstr($error, "Terminal does not support the chosen shipment's size")) {
                        $output .= $this->displayError($this->l("Terminal does not support the chosen shipment's size. Please choose another!"));
                    }
                    else if (!empty($error)) {
                        $output .= $this->displayError($error);
                    }

                    if (!empty($new_shipment_id)) {
                        $this->smarty->assign(array(
                            'parcelSize' => $this->getLpExpressParcelSize($new_shipment_id),
                        ));
                    }
                }
            }

            if ($carrire_obj->external_module_name === 'makecommercelpexpress') {
                $parcelSizes = MakeCommerceLpExpress::getParcelSizes();
                $shipment_id = $this->getShipmentId($order);
                $this->smarty->assign(array(
                    'parcelSize' => (!empty($shipment_id) ? (int)$this->getLpExpressParcelSize($shipment_id) : Configuration::get('MAKECOMMERCE_LP_EXPRESS_LT_TEMPLATE')),
                    'possibleSizes' => $parcelSizes
                ));
            }

            $extra_carrier['id'] = $carrire_obj->id;
            $extra_carrier['name'] = $this->name;

            $address = new Address($order->id_address_delivery);
            $extra_carrier['terminal_id'] = $address->other;

            $country_iso_code = Country::getIsoById($address->id_country);
            $terminals = $this->getTerminals($country_iso_code);

            $is_courier = $order->id_carrier == $this->getCarrierId(true,'courier');
            $this->smarty->assign(array(
                'address_url' => $this->context->link->getAdminLink('AdminAddresses'),
                'carrier_name' => $this->carrier_name,
                'shipment_id' => $shipment_id,
                'trackingUrl' => MakeCommerce::getTrackingUrl($order),
                'terminals' => $terminals,
                'id_address' => $order->id_address_delivery,
                'ajax_url' => $this->context->link->getModuleLink($this->name, 'ajax'),
                'parcel_grouping' => $this->getConfig('parcel_grouping'),
                'carrierid' => $carrire_obj->id,
                'carrier' => $extra_carrier,
                'psversion' => _PS_VERSION_,
                'is_courier' => $is_courier,
                'hide_carrier' => empty($terminals) && _PS_MODE_DEV_, // hide carrier if no terminals found and shop in LIVE mode
            ));

            $this->name = 'makecommerce';

            return $output . $this->display(_PS_MODULE_DIR_ .'makecommerce/makecommerce.php', 'shipping_actions.tpl');
        }
    }

    public function hookActionObjectAddressUpdateAfter($params)
    {
        // check whether order carrier is makecommerce courier
        if ($this->getCarrierId(true, 'courier')  && $this->context->controller->controller_type === 'admin') {
            $order = $this->getOrderByDeliveryAadressId($params['object']->id);
            $this->clearShipmentId($order);
        }
    }

    private function getOrderByDeliveryAadressId($id_delivery_address) {
        $query = new DbQuery();
        $query->select('id_order');
        $query->from('orders');
        $query->where('`id_address_delivery` = "' . $id_delivery_address. '"');
        $orders = Db::getInstance()->executeS($query);

        return new Order($orders[0]['id_order']);
    }

    public function getParcelLabel($order){
        $address = new Address($order->id_address_delivery);
        $customer = new Customer($order->id_customer);
        $makecommerce = new MakeCommerce();

        $label_pgsize= trim(Configuration::get('MAKECOMMERCE_LABEL_PGSIZE', $this->context->language->id));
        $shipment_id = $this->getShipmentId($order);

        if (empty($shipment_id)) {
            $error = $this->createShipments($order);
            if (substr($error,0,4) !=""){
                PrestaShopLogger::addLog(
                    json_encode($error),
                    1,
                    null,
                    'MakeCommerce'
                );
                $this->context->controller->errors[] = $error;
                return $error;
            }
            $shipment_id = $this->getShipmentId($order);
        }

        $credentials = $this->getCredentials();
        $orderData = $this->formatParcelLabelRequest($order);

        // check whether courier is used and add proper credentials
        if ($this->carrier_name === 'DPD' && $this->getCarrierId(true,'courier') === $order->id_carrier) {
            unset($credentials['username'], $credentials['password']);
            $credentials['apiKey'] = MakeCommerce::getConfig('dpd_apikey');
        }

        $request_body = array(
            'credentials' => array($credentials),
            'orders' => array($orderData),
            'printFormat' => $label_pgsize
        );

        // use TMS when enabled for Omniva or DPD
        $makecommerceOmnivaTmsEnabled = MakeCommerce::getConfig('OMNIVA_USE_MK_TMS');
        $makecommerceDpdTmsEnabled = MakeCommerce::getConfig('DPD_USE_MK_TMS');
        if (($this->carrier_name === 'OMNIVA' && $makecommerceOmnivaTmsEnabled || $this->carrier_name === 'DPD' && $makecommerceDpdTmsEnabled)
            && $this->getCarrierId(true,'courier') !== $order->id_carrier
        ) {
            unset($request_body['credentials']);
        }

        $api = $makecommerce->getApi();

        $label = $api->createLabels($request_body);
        if (!isset($label->labelUrl)){
            PrestaShopLogger::addLog(
                json_encode($label),
                1,
                null,
                'MakeCommerce'
            );
            $this->context->controller->errors[] = $label;
            return "Something went wrong, see logs for hints";
        }
        if(isset($label->labelUrl) && $label->labelUrl){
            return $label->labelUrl;
        }
    }

    public function hookDisplayBeforeCarrier($params)
    {
//        if ($params['carrier']['id'] != $this->getCarrierId(true, 'courier')) {
//            return $this->hookDisplayCarrierList(array('id_address' => $params['cart']->id_address_delivery));
//        }
        if (!Module::isEnabled('thecheckout')) {
            return $this->hookDisplayCarrierList(array('id_address' => $params['cart']->id_address_delivery));
        }
    }

    public function hookDisplayHeader($params)
    {
        if (!self::$jsIsIncluded) {
            self::$jsIsIncluded = true;
            $this->context->controller->registerJavascript('modules-makecommerceomniva', 'modules/makecommerce/views/js/carrier_script.js', ['position' => 'bottom', 'priority' => 150]);
        }
    }

    public function hookDisplayCarrierExtraContent($params)
    {
        if (Module::isEnabled('thecheckout')) {
            if ($params['carrier']['id'] != $this->getCarrierId(true, 'courier')) {
                return $this->hookDisplayCarrierList(array('id_address' => $params['cart']->id_address_delivery));
            }
        }
    }

    public function hookDisplayCarrierList($params)
    {
        $id_address = (isset($params['address']) ? $params['address']->id : $params['id_address']);
        $selected_carriers = $this->context->cart->getDeliveryOption(null, true);

        $this->id_carrier = $this->getCarrierId();


        if (!$selected_carriers) {
            $selected_carriers = $this->context->cart->getDeliveryOption(null,false);
        }

        $cookie = false;
        $carrier = new Carrier($this->id_carrier);

        if ($carrier->is_module && $carrier->external_module_name == $this->name) {
            $extra_carrier['id'] = $carrier->id;
            $extra_carrier['name'] = $this->name;
            $cookie = $this->getCookie($id_address);
            $zero_cookie = $this->getCookie(0);

            if ($id_address &&
                $zero_cookie &&
                (!empty($zero_cookie['terminal_id']) || !empty($zero_cookie['group_id']))
            ) {
                $this->saveToCookie($id_address, $zero_cookie);
                $this->saveToCookie(0, false);
            }

            if ($cookie !== false && isset($cookie['terminal_id'])) {
                $extra_carrier['terminal_id'] = $cookie['terminal_id'];
            } else {
                $extra_carrier['terminal_id'] = 0;
            }

            $this->smarty->assign(array(
                'ajax_url' => $this->context->link->getModuleLink($this->name, 'ajax'),
                'one_page_checkout' => (bool)Module::isEnabled('thecheckout')
            ));

            return $this->displayList($extra_carrier, $id_address, $carrier->id);

        }
    }

    public function saveToCookie($id_address, $value)
    {
        $this->context->cookie->{$this->name . '_' . $id_address} = serialize($value);
    }

    /**
     * @param $active
     * @param $carrierType - ( courier, parcel )
     * @return int|mixed
     * @throws PrestaShopDatabaseException
     */
    public function getCarrierId($active = true, $carrierType = 'parcel')
    {
        // get original courier carrier id, prestashop creates new carrier on every carrier data update
        $carrier_reference = MakeCommerce::getConfig('courier_'.$this->getCarrierPrefix());

        $query = new DbQuery();
        $query->select('id_carrier');
        $query->from('carrier');
        $query->where('`is_module` = 1');
        $query->where('`deleted` = 0');

        if ($carrierType === 'parcel') {
            $query->where('`id_reference` != "'.$carrier_reference.'"');
        } elseif ($carrierType === 'courier') {
            $query->where('`id_reference` = "'.$carrier_reference.'"');
        }

        $query->where(sprintf('`external_module_name` = \'%s\'', pSQL($this->name)));
        if ($active) {
            $query->where('`active` = 1');
        }

        $carriers = Db::getInstance()->executeS($query);
        if (!empty($carriers)) {
            return $carriers[0]['id_carrier'];
        }
        return 0;
    }

    public function getCookie($id_address)
    {
        $data = $this->context->cookie->{$this->name . '_' . $id_address};
        if ($data) {
            return Tools::unSerialize($data);
        }
        return false;
    }

    public function displayList($extra_carrier, $id_address, $carrier_id)
    {
        $module_name =  $this->name;
        $this->name = 'makecommerce';

        $terminals = $this->getTerminals();

        $this->smarty->assign(array(
            'terminals' => $terminals,
            'carrier' => $extra_carrier,
            'id_address' => $id_address,
            'module' => $module_name,
            'carrierid' => $carrier_id,
            'psversion' => _PS_VERSION_,
            'hide_carrier' => empty($terminals) && _PS_MODE_DEV_, // hide carrier if no terminals found and shop in LIVE mode
        ));

        if($this->getConfig('parcel_grouping')) {
            $template = $this->display('makecommerce', 'terminals.tpl');
        } else {
            $template = $this->display('makecommerce', 'terminals2.tpl');
        }
        $this->name = $module_name;
        return $template;
    }

    /**
     * Gets carrier parcel terminal list
     * @param $country_iso_code - optional parameter, get parcel list of supplied country iso_code
     * @return array
     */
    public function getTerminals($country_iso_code = null)
    {
        if (is_null($country_iso_code)) {
            $context = Context::getContext();
            $address = new Address($context->cart->id_address_delivery);
            if(isset($address->id_country) && $address->id_country){
                $id_country = $address->id_country;
            }else{
                $id_country = Configuration::get('PS_COUNTRY_DEFAULT');
            }
            $country_iso_code = Country::getIsoById($id_country);
        }

        $terminals_json = Db::getInstance()->getValue(
            'SELECT `carriers_list` FROM `'._DB_PREFIX_.'makecommerce_carriers` WHERE `carrier` = \''.$this->carrier_name.'\''
        );

        $terminal_data = json_decode($terminals_json);

        if (!empty($terminal_data)) {
            if (empty($terminal_data->updated) || $terminal_data->updated + self::CACHE_VALID_TIME < time() || empty($terminal_data->terminals)){
                $terminal_data = null;
            }else{
                $terminals = json_decode(json_encode($terminal_data->terminals), true);
            }
        }


        if (empty($terminal_data)) {

            $this->saveTerminals();

            $terminals_json = Db::getInstance()->getValue(
                'SELECT `carriers_list` FROM `'._DB_PREFIX_.'makecommerce_carriers` WHERE `carrier` = \''.$this->carrier_name.'\''
            );

            if(isset($terminals_json) && $terminals_json){
                $terminal_data = json_decode($terminals_json);
                $terminals = json_decode(json_encode($terminal_data->terminals), true);
            }
        }

        $apt_terminals = array();
        if(isset($terminals) && $terminals){
            foreach($terminals as $terminal){
                if($terminal['carrier'] == $this->carrier_name && $terminal['type'] == 'APT' && $terminal['country'] == $country_iso_code){
                    array_push($apt_terminals, $terminal);
                }
            }
        }

        //Group terminals by city name

        $grouped_terminals = array();
        foreach ($apt_terminals as $terminal) {

            $city = $terminal['city'];
            $grouped_terminals[$city][] = $terminal;

        }

        //Move cities with higher priority top of the list

        if(array_key_exists ('Tartu linn', $grouped_terminals)){
            $grouped_terminals = array('Tartu linn' => $grouped_terminals['Tartu linn']) + $grouped_terminals;
        }
        if(array_key_exists ('Tartu', $grouped_terminals)){
            $grouped_terminals = array('Tartu' => $grouped_terminals['Tartu']) + $grouped_terminals;
        }
        if(array_key_exists ('Tallinn', $grouped_terminals)){
            $grouped_terminals = array('Tallinn' => $grouped_terminals['Tallinn']) + $grouped_terminals;
        }
        return $grouped_terminals;

    }

    public function saveTerminals(){

        $request_body = array(
            'carriers' => [$this->carrier_name],
            'country' => '',
            'type' => ''
        );

        $makecommerce = new MakeCommerce();
        $api = $makecommerce->getApi();

        try{
            $terminals = $api->getDestinations($request_body);
        }catch (Exception $e){
            PrestaShopLogger::addLog(
                $e->getMessage(),
                1,
                null,
                'MakeCommerce'
            );
            return false;
        }

        $object = new stdClass();
        $object->updated = time();
        $object->terminals = $terminals;
        $terminals_json = json_encode($object);

        $exist = Db::getInstance()->getValue(
            'SELECT EXISTS(SELECT `carriers_list` FROM `'._DB_PREFIX_.'makecommerce_carriers` WHERE `carrier` = \''.$this->carrier_name.'\')'
        );

        if ($exist) {
            Db::getInstance()->execute(
                'UPDATE `'._DB_PREFIX_.'makecommerce_carriers` SET `carriers_list` = \''.pSQL($terminals_json).'\' WHERE `carrier` = \''.$this->carrier_name.'\''
            );
        } else {
            Db::getInstance()->execute(
                'INSERT INTO `'._DB_PREFIX_.'makecommerce_carriers` (`carrier`, `carriers_list`) VALUES(\''.$this->carrier_name.'\', \''.pSQL($terminals_json).'\')'
            );
        }

        return true;

    }

    public function createShipments($order)
    {
        $makecommerce = new MakeCommerce();

        $carrier_prefix = $this->getCarrierPrefix();
        $credentials = $this->getCredentials();
        $orders = $this->formatCreateShipmentRequest($order);
        $request_body = array(
            'credentials' => array($credentials),
            'orders' => array($orders)
        );

        // use TMS when enabled for Omniva or DPD
        $makecommerceOmnivaTmsEnabled = MakeCommerce::getConfig('OMNIVA_USE_MK_TMS');
        $makecommerceDpdTmsEnabled = MakeCommerce::getConfig('DPD_USE_MK_TMS');
        if (($this->carrier_name === 'OMNIVA' && $makecommerceOmnivaTmsEnabled || $this->carrier_name === 'DPD' && $makecommerceDpdTmsEnabled)
            && $this->getCarrierId(true,'courier') !== $order->id_carrier
        ) {
            unset($request_body['credentials']);
        }

        // check whether courier is used and add proper credentials
        if ($this->carrier_name === 'DPD' && $this->getCarrierId(true,'courier') === $order->id_carrier) {
            unset($request_body['credentials'][0]['username'], $request_body['credentials'][0]['password']);
            $request_body['credentials'][0]['apiKey'] = MakeCommerce::getConfig('dpd_apikey');
        }

        $api = $makecommerce->getApi();
        try{
            $shipments = $api->createShipments($request_body);
        }catch (Exception $e){

        }

        $shipments = json_decode(json_encode($shipments), true);

        if (isset($shipments[0]["errorMessage"])) {
            PrestaShopLogger::addLog(
                $shipments[0]["errorMessage"],
                1,
                null,
                'MakeCommerce'
            );
            return $shipments[0]["errorMessage"];
        } elseif (isset($shipments['shipments'][0]['errorMessage'])) {
            return $shipments['shipments'][0]['errorMessage'];
        }

        if (isset($shipments[0]['shipmentId']) || isset($shipments['shipments'][0]['shipmentId']))
        {
            // tracking number for lpexpress
            if ($this->name === 'makecommercelpexpress' && isset($shipments[0]['barcode'])) {
                $trackingNumber = $shipments[0]['barcode'];
            }
            elseif (isset($shipments[0]['shipmentId'])) {
                $trackingNumber = $shipments[0]['shipmentId'];
            }
            // Omniva courier tracking number
            elseif (isset($shipments['shipments'][0]['shipmentId'])) {
                $trackingNumber = ($shipments['shipments'][0]['shipmentId']);
            }

            Db::getInstance()->update(
                'order_carrier',
                array(
                    'tracking_number' => pSQL($trackingNumber)
                ),
                '`id_order`=' . $order->id
            );

            // LP_EXPRESS_LT unique field
            if (isset($shipments[0]['lpExpressCartIdentifier']))
            {
                $parcelSize = Tools::getValue('parcelSize') ?: MakeCommerce::getConfig($carrier_prefix.'_template');

                Db::getInstance()->insert(
                    'makecommerce_lpexpress',
                    array(
                        'shipmentId' => pSQL($shipments[0]['shipmentId']),
                        'trackingNumber' => pSQL($trackingNumber),
                        'lpExpressCartIdentifier' => pSQL($shipments[0]['lpExpressCartIdentifier']),
                        'parcelSize' => pSQL($parcelSize)
                    )
                );
            }
        }
    }

    public function formatParcelLabelRequest(Order $order)
    {
        $address = new Address($order->id_address_delivery);
        $carrier_prefix = $this->getCarrierPrefix();
        $shipment_id = $this->getShipmentId($order);
        $customer = new Customer($order->id_customer);
        // Prepare phone number - for Omniva mainly
        $phone_number = (isset($address->phone_mobile) && $address->phone_mobile) ? $address->phone_mobile : $address->phone;
        $phone_number = $this->formatPhoneNumber($phone_number, $address->id_country);

        if (in_array($carrier_prefix, ['OMNIVA_LV', 'OMNIVA_LT'])) {
            $carrier_name = $carrier_prefix;
            $carrier_prefix = $this->carrier_name;
        } else {
            $carrier_name = $this->carrier_name;
        }

        $orderData = array(
            'orderId' => 'order '.$order->id,
            'carrier' => $carrier_name,
            'recipient' => array(
                'name' => $address->firstname.' '.$address->lastname,
                'phone' => $phone_number, // (isset($address->phone_mobile) && $address->phone_mobile) ? $address->phone_mobile : $address->phone,
                'email' => $customer->email
            ),
            'sender' => array(
                'name' => MakeCommerce::getConfig($carrier_prefix.'_sender_name'),
                'phone' => MakeCommerce::getConfig($carrier_prefix.'_phone'),
                'email' => MakeCommerce::getConfig($carrier_prefix.'_email'),
                'street' => MakeCommerce::getConfig($carrier_prefix.'_street'),
                'city' => MakeCommerce::getConfig($carrier_prefix.'_city'),
                'country' => MakeCommerce::getConfig($carrier_prefix.'_country'),
                'postalCode' => $this->formatPostCode(MakeCommerce::getConfig($carrier_prefix.'_zip'), $address->id_country),
            ),
            'shipmentId' => $shipment_id
        );

        /**
         * if no terminal id present (stored in address->other field) and carrier name contains courier
         * add destination data as an address, otherwise chosen terminal ID is submitted
         */
        if ($this->getCarrierId(true,'courier') === $order->id_carrier) {
            /**
             * Omniva courier adds additional data to the shipment request
             */
            if (strstr(strtolower($this->carrier_name), 'omniva')) {
                $orderData['services'] = [ "serviceType" => "PK"];
            }
            $orderData['destination'] = array(
                "postalCode" => $address->postcode,
                "country" => Country::getIsoById($address->id_country),
                "city" => $address->city,
                "street" => $address->address1
            );
        } else {
            $orderData['destination'] = ['destinationId' => $address->other];
        }

        if ($this->carrier_name === 'LP_EXPRESS_LT') {
            $orderData['sender']['country'] = 'LT';
            $orderData['lpExpressShipmentDetails']['lpExpressCartIdentifier'] = Db::getInstance()->getValue('SELECT `lpExpressCartIdentifier` FROM `'._DB_PREFIX_.'makecommerce_lpexpress` WHERE `shipmentId` = "' . $shipment_id . '"');
            $orderData['destination']['postalCode'] = $this->formatPostCode($address->postcode, $address->id_country);
            $orderData['destination']['country'] = 'LT';
            $orderData['destination']['city'] = $address->city;
            $orderData['destination']['street'] = $address->address1;
        }

        return $orderData;
    }

    public function formatCreateShipmentRequest(Order $order)
    {
        $address = new Address($order->id_address_delivery);
        $carrier_prefix = $this->getCarrierPrefix();
        $shipment_id = $this->getShipmentId($order);
        $customer = new Customer($order->id_customer);
        // Prepare phone number - for Omniva mainly
        $phone_number = (isset($address->phone_mobile) && $address->phone_mobile) ? $address->phone_mobile : $address->phone;
        $phone_number = $this->formatPhoneNumber($phone_number, $address->id_country);

        if (in_array($carrier_prefix, ['OMNIVA_LV', 'OMNIVA_LT'])) {
            $carrier_name = $carrier_prefix;
            $carrier_prefix = $this->carrier_name;
        } else {
            $carrier_name = $this->carrier_name;
        }
        $orderData = array(
            'orderId'=> 'order '.$order->id,
            'carrier'=> $carrier_name,
            'recipient'=> array(
                'name' => $address->firstname.' '.$address->lastname,
                'phone' => $phone_number, // (isset($address->phone_mobile) && $address->phone_mobile) ? $address->phone_mobile : $address->phone,
                'email' => $customer->email,
            ),
            'sender' => array(
                'name' => MakeCommerce::getConfig($carrier_prefix.'_sender_name'),
                'phone' => MakeCommerce::getConfig($carrier_prefix.'_phone'),
                'email' => MakeCommerce::getConfig($carrier_prefix.'_email'),
                'street' => MakeCommerce::getConfig($carrier_prefix.'_street'),
                'city' => MakeCommerce::getConfig($carrier_prefix.'_city'),
                'country' => MakeCommerce::getConfig($carrier_prefix.'_country'),
                'postalCode' => MakeCommerce::getConfig($carrier_prefix.'_zip')
            )
        );

        /**
         * if no terminal id present (stored in address->other field) and carrier name contains courier
         * add destination data as an address, otherwise chosen terminal ID is submitted
         */
        $id_courier = $this->getCarrierId(true,'courier');
        if (!empty($id_courier) && $id_courier === $order->id_carrier) {
            /**
             * Omniva courier adds additional data to the shipment request
             */
            if (strstr(strtolower($this->carrier_name), 'omniva')) {
                $orderData['services'] = [ "serviceType" => "PK"];
            }
            $orderData['destination'] = array(
                "postalCode" => $address->postcode,
                "country" => Country::getIsoById($address->id_country),
                "city" => $address->city,
                "street" => $address->address1
            );
        } else {
            $orderData['destination'] = ['destinationId' => $address->other];
        }

        // LP express specific fields
        if ($this->carrier_name === 'LP_EXPRESS_LT') {
            $parcelSize = Tools::getValue('parcelSize') ?: MakeCommerce::getConfig($carrier_prefix.'_template');
            $orderData['destination']['postalCode'] = $this->formatPostCode($address->postcode, $address->id_country);
            $orderData['destination']['country'] = 'LT';
            $orderData['destination']['city'] = $address->city;
            $orderData['destination']['street'] = $address->address1;

            $orderData['sender']['country'] = Country::getIsoById($address->id_country);
            $orderData['sender']['building'] = MakeCommerce::getConfig($carrier_prefix.'_building');
            $orderData['sender']['postalCode'] = $this->formatPostCode(MakeCommerce::getConfig($carrier_prefix.'_zip'), $address->id_country);
            $orderData['lpExpressShipmentDetails']['templateId'] = $parcelSize;
        }

        return $orderData;
    }

    public function getCredentials()
    {
        $carrier_prefix = $carrier = $this->getCarrierPrefix();
        $credentials = array(
            'carrier' => $carrier_prefix,
            'username' => MakeCommerce::getConfig($this->carrier_name.'_username'),
            'password' => MakeCommerce::getConfig($this->carrier_name.'_password')
        );

        if ($this->carrier_name === 'SMARTPOST') {
            $credentials['apiKey'] = MakeCommerce::getConfig($this->carrier_name.'_apikey');
        }

        return $credentials;
    }

    private function getLpExpressParcelSize($shipmentid)
    {
        return Db::getInstance()->getValue('SELECT `parcelSize` FROM `'._DB_PREFIX_.'makecommerce_lpexpress` WHERE `shipmentId` = ' . $shipmentid);
    }

    public function getShipmentId(Order $order)
    {
        // LP express shipment id differs from tracking number
        if ($this->name === 'makecommercelpexpress') {
            $trackingNumber = Db::getInstance()->getValue(
                'SELECT `shipmentId` FROM `'._DB_PREFIX_.'makecommerce_lpexpress` 
                WHERE `trackingNumber` = (SELECT `tracking_number` FROM `'._DB_PREFIX_.'order_carrier` WHERE `id_order`= ' . $order->id . ' LIMIT 1)'
            );
        } else {
            $trackingNumber = Db::getInstance()->getValue(
                'SELECT `tracking_number` FROM `'._DB_PREFIX_.'order_carrier` WHERE `id_order`=' . $order->id
            );
        }

        return $trackingNumber;
    }

    public function clearShipmentId(Order $order)
    {
        return Db::getInstance()->update('order_carrier',
            ['tracking_number' => null],
            '`id_order`=' . $order->id
        );
    }

    public function getCarrierPrefix()
    {
        if ($this->carrier_name == 'OMNIVA') {
            $carrier_prefix = trim(Configuration::get('MAKECOMMERCE_OMNIVA_COUNTRYSEL', $this->context->language->id));
        } elseif ($this->carrier_name == 'DPD') {
            $carrier_prefix = trim(Configuration::get('MAKECOMMERCE_DPD_COUNTRYSEL', $this->context->language->id));
        } elseif ($this->carrier_name == 'LPEXPRESS') {
            $carrier_prefix = 'lpexpress';
        } else {
            $carrier_prefix = $this->carrier_name;
        }

        return $carrier_prefix;
    }

    private function formatPostCode($postCode, $country_id)
    {
        switch (Country::getIsoById($country_id)) {
            case 'LT':
                if (substr($postCode, 0, 3) === 'LT-') {
                    $postCode = substr($postCode, 3);
                }
                break;
        }

        return $postCode;
    }

    // check phone number formatting and if necessary add reginal code or '+' in front
    private function formatPhoneNumber($phone_number, $country_id)
    {
        if( substr( $phone_number, 0, 1) != '+') {
            switch(Country::getIsoById($country_id)) {
                case 'EE':
                    if (substr($phone_number,0,1) == '5')
                        $phone_number = '+372' . $phone_number;
                    elseif (substr($phone_number,0,4) == '3725')
                        $phone_number = '+' . $phone_number;
                    elseif (substr($phone_number,0,5) == '+3725')
                        ; // break;
                    else
                        error_log("Recipients phone number $phone_number is (probably) invalid for country Estonia!");
                    break;
                case 'LV':
                    if (substr($phone_number,0,1) == '2')
                        $phone_number = '+371' . $phone_number;
                    elseif (substr($phone_number,0,4) == '3712')
                        $phone_number = '+' . $phone_number;
                    elseif (substr($phone_number,0,5) == '+3712')
                        ; // break;
                    else
                        error_log("Recipients phone number $phone_number is (probably) invalid for country Latvia!");
                    break;
                case 'LT':
                    if (substr($phone_number,0,1) == '6')
                        $phone_number = '+370' . $phone_number;
                    elseif (substr($phone_number,0,4) == '3706')
                        $phone_number = '+' . $phone_number;
                    elseif (substr($phone_number,0,5) == '+3706')
                        ; // break;
                    else
                        error_log("Recipients phone number $phone_number is (probably) invalid for country Lithuania!");
            }
        }

        return $phone_number;
    }
}
